import {
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   IModalProps,
   isEmpty,
   Loader,
   ModalLayout,
   parseApiError,
   toasts,
   useBehaviourSubject,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import { Button } from 'lego-on-react';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import {
   AutomationType,
   dnsAutomationCreditsFormValidationSchema,
   globalState,
   IApiError,
   IAutomationCreditsForm,
   IDnsAutomationCreditsForm,
} from '../../models';
import { auth, projectApi } from '../../services';

import classes from '../../design/commonModals.module.css';
import { CreditField } from './components/CreditField';
import { DnsDomainField } from './components/DnsDomainField';
import { ReasonField } from '../form_fields2/ReasonField';

const initialValue: IDnsAutomationCreditsForm = {
   credit: {
      time: 0,
   },
   dns_domain: '',
   reason: '',
};

interface InternalFormProps {
   /**
    * Только админы могут редактировать dns_domain
    */
   canEditDnsDomain: boolean;

   /**
    * Блокирует включение автоматики, если dns_domain не задан
    */
   locked: boolean;
}

const InternalForm: React.FC<PropsWithChildren<InternalFormProps>> = ({ canEditDnsDomain, locked, children }) => (
   <Form className={classes.form}>
      <DnsDomainField
         name={'dns_domain'}
         help={canEditDnsDomain ? '' : 'Contact to admin to change the domain'}
         readonly={!canEditDnsDomain}
      />

      <CreditField name={'credit'} disabled={locked} controlProps={{ automationType: AutomationType.DNS }} />

      <ReasonField disabled={locked} />

      {children}
   </Form>
);

interface Props extends IModalProps<IAutomationCreditsForm> {
   dnsDomain: string;
   id: string;
}

export const DnsOnModal: React.FC<Props> = ({ dnsDomain, id, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const user = useBehaviourSubject(globalState.user);
   const canEditDnsDomain = auth.canEditDnsDomain(user);

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<IDnsAutomationCreditsForm>),
      [],
   );

   const initialValuePatched = useMemo(() => ({ ...initialValue, dns_domain: dnsDomain }), [dnsDomain]);

   const enableAutomation = useCallback(
      (value: IAutomationCreditsForm, helpers: FormikHelpers<IDnsAutomationCreditsForm>) => {
         projectApi
            .enableAutomation(AutomationType.DNS, id, value)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  ok(value);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok, id],
   );

   const handleSubmit = useCallback(
      (v: IDnsAutomationCreditsForm, helpers: FormikHelpers<IDnsAutomationCreditsForm>) => {
         const { dns_domain, ...value } = v;
         if (dnsDomain === dns_domain) {
            enableAutomation(value, helpers);

            return;
         }

         projectApi
            .updateDnsDomain(id, dns_domain, v.reason)
            .pipe(takeUntil(dismounted))
            .subscribe(() => {
               if (!isEmpty(dns_domain)) {
                  enableAutomation(value, helpers);
               }
            }, toasts.handleApiError('DNS-domain updating'));
      },
      [dismounted, dnsDomain, enableAutomation, id],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} showFooter={false} title={'You are going to turn on the DNS-automation'}>
         <Formik
            initialValues={initialValuePatched}
            onSubmit={handleSubmit}
            validationSchema={dnsAutomationCreditsFormValidationSchema}
         >
            {form => {
               const locked = isEmpty(form.values.dns_domain);

               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <InternalForm canEditDnsDomain={canEditDnsDomain} locked={locked}>
                        <footer className={classes.footer}>
                           {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                           <Loader visible={form.isSubmitting} inline={true} text={'DNS is turning on'} />

                           <Button
                              theme={'clear'}
                              size={'s'}
                              cls={classes.button}
                              onClick={dismiss}
                              disabled={form.isSubmitting}
                           >
                              Cancel
                           </Button>

                           <Button
                              theme={'action'}
                              size={'s'}
                              cls={classes.button}
                              type={'submit'}
                              disabled={!form.isValid || form.isSubmitting || (!canEditDnsDomain && locked)}
                           >
                              Turn on
                           </Button>
                        </footer>
                     </InternalForm>
                  </FormHooksContext.Provider>
               );
            }}
         </Formik>
      </ModalLayout>
   );
};

DnsOnModal.displayName = 'DnsOnModal';
