import {
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import { Button } from 'lego-on-react';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { automationFormValidationSchema, IApiError, IAutomationForm } from '../../models';
import { projectApi } from '../../services';

import classes from '../../design/commonModals.module.css';
import { ReasonField } from '../form_fields2/ReasonField';

const initialValue: IAutomationForm = {
   reason: '',
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <ReasonField />

      {children}
   </Form>
);

interface Props extends IModalProps<IAutomationForm> {
   id: string;
}

export const FsmOffModal: React.FC<Props> = ({ id, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<IAutomationForm>),
      [],
   );

   const handleSubmit = useCallback(
      (v: IAutomationForm, helpers: FormikHelpers<IAutomationForm>) => {
         projectApi
            .disableFsmHandbrake(id, v.reason)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  ok(v);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok, id],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} showFooter={false} title={'You are going to turn off the FSM-handbrake'}>
         <Formik initialValues={initialValue} onSubmit={handleSubmit} validationSchema={automationFormValidationSchema}>
            {form => (
               <FormHooksContext.Provider value={formHooks}>
                  <InternalForm>
                     <footer className={classes.footer}>
                        {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                        <Loader visible={form.isSubmitting} inline={true} text={'FSM-handbrake is turning off'} />

                        <Button
                           theme={'clear'}
                           size={'s'}
                           cls={classes.button}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                        >
                           Cancel
                        </Button>

                        <Button
                           theme={'action'}
                           size={'s'}
                           cls={classes.button}
                           type={'submit'}
                           disabled={!form.isValid || form.isSubmitting}
                        >
                           Turn off
                        </Button>
                     </footer>
                  </InternalForm>
               </FormHooksContext.Provider>
            )}
         </Formik>
      </ModalLayout>
   );
};

FsmOffModal.displayName = 'FsmOffModal';
