import {
   classNames,
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   fromTimestamp,
   HOUR_IN_SEC,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { addSeconds, differenceInSeconds } from 'date-fns';
import { Form, Formik, FormikHelpers } from 'formik';
import { Button } from 'lego-on-react';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import classes from '../../design/commonModals.module.css';

import {
   fsmHandbrakeFormValidationSchema,
   FsmHandbrakePayload,
   IApiError,
   IFsmHandbrakeForm,
   IFsmHandbrakeProjectParams,
   TimeoutMode,
} from '../../models';
import { projectApi } from '../../services';
import { ReasonField } from '../form_fields2/ReasonField';
import { TimeoutField } from '../form_fields2/TimeoutField';
import { TrackerIssueField } from '../form_fields2/TrackerIssueField';

const initialValue: IFsmHandbrakeForm = {
   reason: '',
   ticket_key: '',
   timeout: {
      date: addSeconds(new Date(), HOUR_IN_SEC),
      mode: TimeoutMode.Date,
      timer: HOUR_IN_SEC,
   },
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <TrackerIssueField name={'ticket_key'} label={'Ticket'} />

      <TimeoutField name={'timeout'} label={'Deactivation date'} />

      <ReasonField />

      {children}
   </Form>
);

interface Props extends IModalProps<IFsmHandbrakeForm> {
   details?: FsmHandbrakePayload;
   id: string;
}

export const FsmOnModal: React.FC<Props> = ({ id, details, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const initialValuePatched: IFsmHandbrakeForm = useMemo(() => {
      if (!details) {
         return initialValue;
      }

      const now = new Date();
      const date = fromTimestamp(details.timeout_time);

      return {
         ...initialValue,
         ticket_key: details.ticket_key || '',
         timeout: {
            date,
            mode: TimeoutMode.Date,
            timer: differenceInSeconds(date, now),
         },
      };
   }, [details]);

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<IFsmHandbrakeForm>),
      [],
   );

   const handleSubmit = useCallback(
      (v: IFsmHandbrakeForm, helpers: FormikHelpers<IFsmHandbrakeForm>) => {
         const params: IFsmHandbrakeProjectParams = {
            _mode: v.timeout.mode,
            reason: v.reason,
            ticket_key: v.ticket_key!,
            timeout: v.timeout.timer,
            timeout_time: v.timeout.date ?? undefined,
         };

         projectApi
            .enableFsmHandbrake(id, params)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  ok(v);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok, id],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const subtitle = (
      <p>
         I will be impossible to detect problems for these hosts.
         <br />
         Their instances will not be relocated automatically.
      </p>
   );

   return (
      <ModalLayout
         onDismiss={dismiss}
         showFooter={false}
         title={details ? 'You are going to renew the FSM-handbrake' : 'You are going to turn on the FSM-handbrake'}
         subtitle={subtitle}
      >
         <Formik
            initialValues={initialValuePatched}
            onSubmit={handleSubmit}
            validationSchema={fsmHandbrakeFormValidationSchema}
         >
            {form => (
               <FormHooksContext.Provider value={formHooks}>
                  <InternalForm>
                     <footer className={classes.footer}>
                        {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                        <Loader visible={form.isSubmitting} inline={true} text={'FSM-handbrake is turning on'} />

                        <Button
                           theme={'clear'}
                           size={'s'}
                           cls={classes.button}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                        >
                           Cancel
                        </Button>

                        <Button
                           theme={'action'}
                           size={'s'}
                           cls={classNames(classes.button, classes.danderButton)}
                           type={'submit'}
                           disabled={!form.isValid || form.isSubmitting}
                        >
                           Turn on
                        </Button>
                     </footer>
                  </InternalForm>
               </FormHooksContext.Provider>
            )}
         </Formik>
      </ModalLayout>
   );
};

FsmOnModal.displayName = 'FsmOnModal';
