import {
   classNames,
   DISMISS_REASON,
   formatDate,
   FormHooks,
   FormHooksContext,
   IModalProps,
   InputField2,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import { Button } from 'lego-on-react';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';
import classes from '../../design/commonModals.module.css';

import {
   Automation,
   automationFormValidationSchema,
   AutomationType,
   HealingAutomationForm,
   IApiError,
   IAutomationForm,
} from '../../models';
import { projectApi } from '../../services';
import { ReasonField } from '../form_fields2/ReasonField';

const initialValue: HealingAutomationForm = {
   reason: '',
   credit_end_time: '',
   credit: {},
};

interface InternalFormProps {
   healing: Automation;
}

const InternalForm: React.FC<PropsWithChildren<InternalFormProps>> = ({ children, healing }) => (
   <Form>
      <InputField2 name={'credit_end_time'} label={'Credit end time'} readonly={true} />

      {healing.credit
         ? Object.keys(healing.credit).map(k => <InputField2 key={k} name={`credit.${k}`} label={k} readonly={true} />)
         : null}

      <ReasonField />

      {children}
   </Form>
);

interface Props extends IModalProps<IAutomationForm> {
   healingAutomation: Automation;
   id: string;
}

export const HealingOffModal: React.FC<Props> = ({ id, healingAutomation, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<IAutomationForm>),
      [],
   );

   const toDateString = (n: number) => formatDate(new Date(n * 1000));

   const initialValuePatched = useMemo(
      () => ({
         ...initialValue,
         credit: healingAutomation.credit!,
         credit_end_time: healingAutomation.credit_end_time ? toDateString(healingAutomation.credit_end_time) : '',
      }),
      [healingAutomation],
   );

   const handleSubmit = useCallback(
      (v: HealingAutomationForm, helpers: FormikHelpers<HealingAutomationForm>) => {
         projectApi
            .disableAutomation(AutomationType.Healing, id, v.reason)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  ok(v);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok, id],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const subtitle = (
      <p>
         I will be impossible to detect health problems for these hosts.
         <br />
         Their instances will not be relocated automatically.
      </p>
   );

   return (
      <ModalLayout
         onDismiss={dismiss}
         showFooter={false}
         title={'You are going to turn off the Healing automation'}
         subtitle={subtitle}
      >
         <Formik
            initialValues={initialValuePatched}
            onSubmit={handleSubmit}
            validationSchema={automationFormValidationSchema}
         >
            {form => (
               <FormHooksContext.Provider value={formHooks}>
                  <InternalForm healing={healingAutomation}>
                     <footer className={classes.footer}>
                        {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                        <Loader visible={form.isSubmitting} inline={true} text={'Healing is turning off'} />

                        <Button
                           theme={'clear'}
                           size={'s'}
                           cls={classes.button}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                        >
                           Cancel
                        </Button>

                        <Button
                           theme={'action'}
                           size={'s'}
                           cls={classNames(classes.button, classes.danderButton)}
                           type={'submit'}
                           disabled={!form.isValid || form.isSubmitting}
                        >
                           Turn off
                        </Button>
                     </footer>
                  </InternalForm>
               </FormHooksContext.Provider>
            )}
         </Formik>
      </ModalLayout>
   );
};

HealingOffModal.displayName = 'HealingOffModal';
