import {
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import { Button } from 'lego-on-react';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { automationCreditsFormValidationSchema, AutomationType, IApiError, IAutomationCreditsForm } from '../../models';
import { projectApi } from '../../services';
import classes from '../../design/commonModals.module.css';

import { CreditField } from './components/CreditField';
import { ReasonField } from '../form_fields2/ReasonField';

const initialValue: IAutomationCreditsForm = {
   credit: {
      time: 0,
   },
   reason: '',
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <CreditField name={'credit'} controlProps={{ automationType: AutomationType.Healing }} />

      <ReasonField />

      {children}
   </Form>
);

interface Props extends IModalProps<IAutomationCreditsForm> {
   id: string;
}

export const HealingOnModal: React.FC<Props> = ({ id, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<IAutomationCreditsForm>),
      [],
   );

   const handleSubmit = useCallback(
      (v: IAutomationCreditsForm, helpers: FormikHelpers<IAutomationCreditsForm>) => {
         projectApi
            .enableAutomation(AutomationType.Healing, id, v)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  ok(v);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok, id],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} showFooter={false} title={'You are going to turn on the Healing automation'}>
         <Formik
            initialValues={initialValue}
            onSubmit={handleSubmit}
            validationSchema={automationCreditsFormValidationSchema}
         >
            {form => (
               <FormHooksContext.Provider value={formHooks}>
                  <InternalForm>
                     <footer className={classes.footer}>
                        {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                        <Loader visible={form.isSubmitting} inline={true} text={'Healing is turning on'} />

                        <Button
                           theme={'clear'}
                           size={'s'}
                           cls={classes.button}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                        >
                           Cancel
                        </Button>

                        <Button
                           theme={'action'}
                           size={'s'}
                           cls={classes.button}
                           type={'submit'}
                           disabled={!form.isValid || form.isSubmitting}
                        >
                           Turn on
                        </Button>
                     </footer>
                  </InternalForm>
               </FormHooksContext.Provider>
            )}
         </Formik>
      </ModalLayout>
   );
};

HealingOnModal.displayName = 'HealingOnModal';
