import { EMPTY_VALUE, formatDate, fromTimestamp, styleHelpers, useBehaviourSubject } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useMemo } from 'react';
import { Link } from 'react-router-dom';
import { DnsZoneFormParams } from 'state/fullProjectForm';

import { ProjectItem } from 'state/projects/models';
import { auditLogTypesForAutomations, FsmHandbrakePayload, globalState, ROUTE_LINKS, Automation } from '../../models';
import { auth } from '../../services';
import { BiasedTumbler, FadeButton, useModals } from '../../shared';
import { SimpleFieldLayout } from '../form_fields2/SimpleFieldLayout';
import { ProjectWarnings2 } from '../ProjectSelect2/components/ProjectWarnings2';
import { AutomationModals, modalConfigs } from './configs';
import classes from './ProjectAutomationTumblers.module.css';

import { PencilIcon } from 'design/icons';

interface Props {
   disabled: boolean;
   dnsAutomation: Automation;
   dnsDomain?: string;
   dnsZone?: DnsZoneFormParams;
   fsmDetails?: FsmHandbrakePayload;
   fsmEnabled: boolean;
   healingAutomation: Automation;
   projectId: string;
   projectName: string;

   onChange(): void;
}

export const ProjectAutomationTumblers: React.FC<Props> = React.memo(
   ({
      projectId,
      projectName,
      disabled,
      dnsAutomation,
      dnsDomain,
      dnsZone,
      fsmDetails,
      fsmEnabled,
      healingAutomation,
      onChange,
   }) => {
      const user = useBehaviourSubject(globalState.user);
      const canEditDnsDomain = auth.canEditDnsDomain(user);
      const project: ProjectItem[] = useMemo(() => {
         return [
            {
               dns_automation: dnsAutomation,
               dns_domain: dnsDomain,
               yc_dns_zone_id: dnsZone?.dnsZoneID,
               fsm_handbrake: fsmDetails,
               healing_automation: healingAutomation,
               id: projectId,
               name: projectName,
            },
         ];
      }, [dnsAutomation, dnsDomain, dnsZone, fsmDetails, healingAutomation, projectId, projectName]);

      const { submitting, openModal } = useModals(modalConfigs, onChange);

      //region handlers
      const handleHealingChange = useCallback(
         () => openModal(projectId, AutomationModals.Healing, !healingAutomation.enabled, { healingAutomation }),
         [healingAutomation, openModal, projectId],
      );

      const handleDnsChange = useCallback(
         () => openModal(projectId, AutomationModals.DNS, !dnsAutomation.enabled, { dnsAutomation, dnsDomain }),
         [openModal, projectId, dnsAutomation, dnsDomain],
      );

      const handleDnsEdit = useCallback(
         () =>
            openModal(projectId, AutomationModals.DnsDomain, !dnsAutomation.enabled, {
               dnsDomain,
               dnsZoneId: dnsZone?.dnsZoneID,
            }),
         [openModal, projectId, dnsAutomation, dnsDomain, dnsZone],
      );

      const handleFsmChange = useCallback(
         () => openModal(projectId, AutomationModals.FSM, !fsmEnabled, { fsmDetails }),
         [fsmDetails, fsmEnabled, openModal, projectId],
      );

      const handleFsmRenew = useCallback(
         () => openModal(projectId, AutomationModals.FsmRenew, !fsmEnabled, { fsmDetails }),
         [fsmDetails, fsmEnabled, openModal, projectId],
      );
      //endregion

      return (
         <ProjectWarnings2 projects={project}>
            <SimpleFieldLayout label={'Healing-automation'} id={'healing'}>
               <BiasedTumbler
                  value={healingAutomation.enabled}
                  onChange={handleHealingChange}
                  disabled={disabled || submitting}
                  preferredValue={true}
                  trueLabel={'Turned on'}
                  falseLabel={'Turned off'}
               />
            </SimpleFieldLayout>

            <SimpleFieldLayout label={'DNS-automation'} id={'dns'} classNameInput={classes.dnsField}>
               <BiasedTumbler
                  value={dnsAutomation.enabled}
                  onChange={handleDnsChange}
                  disabled={disabled || submitting}
                  preferredValue={true}
                  trueLabel={'Turned on'}
                  falseLabel={'Turned off'}
               />

               {canEditDnsDomain ? (
                  <>
                     <div className={classes.inputAdditional}>
                        DNS-domain: {dnsDomain || EMPTY_VALUE}
                        <br />
                        <div>
                           YC DNS-zone:{' '}
                           {dnsZone?.domain ? (
                              <a
                                 href={dnsZone.link}
                                 className={styleHelpers.externalLink}
                                 target={'_blank'}
                                 rel={'noopener noreferrer'}
                              >
                                 {dnsZone.domain}
                              </a>
                           ) : (
                              EMPTY_VALUE
                           )}
                        </div>
                     </div>
                     <FadeButton text={'Edit'} onClick={handleDnsEdit} icon={<PencilIcon />} />
                  </>
               ) : (
                  <div className={classes.inputAdditional}>
                     {dnsDomain ? `DNS-domain: ${dnsDomain}` : null}
                     <br />
                     {dnsZone?.domain ? (
                        <>
                           YC DNS-zone:{' '}
                           <a
                              href={dnsZone.link}
                              className={styleHelpers.externalLink}
                              target={'_blank'}
                              rel={'noopener noreferrer'}
                           >
                              {dnsZone.domain}
                           </a>
                        </>
                     ) : null}
                  </div>
               )}
            </SimpleFieldLayout>

            <SimpleFieldLayout label={'FSM-handbrake'} id={'fsm'}>
               <BiasedTumbler
                  value={fsmEnabled}
                  onChange={handleFsmChange}
                  disabled={disabled || submitting}
                  preferredValue={false}
                  trueLabel={'Turned on'}
                  falseLabel={'Turned off'}
               />

               {fsmEnabled && fsmDetails ? (
                  <div className={classes.inputAdditional}>
                     Deactivation date: {formatDate(fromTimestamp(fsmDetails.timeout_time!), 'd MMM yyyy, HH:mm')}
                     <FadeButton text={'Edit'} onClick={handleFsmRenew} icon={<PencilIcon />} />
                  </div>
               ) : null}
            </SimpleFieldLayout>

            <Link
               to={ROUTE_LINKS.auditLog({
                  project: projectId,
                  type: auditLogTypesForAutomations,
               })}
            >
               View audit log for automations
            </Link>
         </ProjectWarnings2>
      );
   },
);

ProjectAutomationTumblers.displayName = 'ProjectAutomationTumblers';
