import { useBehaviourSubject } from '@yandex-infracloud-ui/libs';
import React, { useCallback } from 'react';
import { Link } from 'react-router-dom';
import { getTumblerValue, globalState, ROUTE_LINKS } from '../../models';
import { auth } from '../../services';
import { BiasedTumbler, IsActualWrapper, useModals } from '../../shared';
import { ProjectItem } from '../../state/projects/models';
import { isActual } from '../../utils/toLibs';
import { AutomationModals, modalConfigs } from './configs';

import classes from './ProjectListAutomationTumblers.module.css';

interface Props {
   items: ProjectItem[];

   onChange(projectId: string): void;
}

export const ProjectListAutomationTumblers: React.FC<Props> = React.memo(({ items, onChange }) => {
   const user = useBehaviourSubject(globalState.user);
   const { submitting, openModal } = useModals(modalConfigs, onChange);

   //region handlers
   const handleHealingChange = useCallback(
      (item: ProjectItem) => openModal(item.id, AutomationModals.Healing, !getTumblerValue(item.healing_automation)),
      [openModal],
   );

   const handleDnsChange = useCallback(
      (item: ProjectItem) =>
         openModal(item.id, AutomationModals.DNS, !getTumblerValue(item.dns_automation), {
            dnsDomain: item.dns_domain,
         }),
      [openModal],
   );

   const handleFsmChange = useCallback(
      (item: ProjectItem) =>
         openModal(
            item.id,
            AutomationModals.FSM,
            item.fsm_handbrake ? !isActual(item.fsm_handbrake.timeout_time) : true,
            { fsmDetails: item.fsm_handbrake },
         ),
      [openModal],
   );
   //endregion

   return (
      <div className={classes.grid}>
         <div className={classes.header}>Project name</div>
         <div className={classes.header}>
            Healing
            <br />
            Automation
         </div>
         <div className={classes.header}>
            DNS-
            <br />
            Automation
         </div>
         <div className={classes.header}>
            FSM-
            <br />
            Handbrake
         </div>
         <div className={classes.separator} />

         {items.map(item => {
            const canEdit = auth.canEditProject({ id: item.id }, user);

            return (
               <React.Fragment key={item.id}>
                  <div className={classes.name}>
                     <Link to={ROUTE_LINKS.project(item.id)}>{item.name}</Link>
                  </div>

                  {/* Healing */}
                  <div className={classes.tumbler}>
                     <BiasedTumbler
                        preferredValue={true}
                        value={item.healing_automation!.enabled}
                        onChange={() => handleHealingChange(item)}
                        disabled={!canEdit || submitting}
                     />
                  </div>

                  {/* DNS */}
                  <div className={classes.tumbler}>
                     <BiasedTumbler
                        preferredValue={true}
                        value={item.dns_automation!.enabled}
                        onChange={() => handleDnsChange(item)}
                        disabled={!canEdit || submitting}
                     />
                  </div>

                  {/* FSM-Handbrake */}
                  <div className={classes.tumbler}>
                     {item.fsm_handbrake ? (
                        <IsActualWrapper timeout={item.fsm_handbrake.timeout_time}>
                           {isFsmActual => (
                              <BiasedTumbler
                                 preferredValue={false}
                                 value={isFsmActual}
                                 onChange={() => handleFsmChange(item)}
                                 disabled={!canEdit || submitting}
                              />
                           )}
                        </IsActualWrapper>
                     ) : (
                        <BiasedTumbler
                           preferredValue={false}
                           value={false}
                           onChange={() => handleFsmChange(item)}
                           disabled={!canEdit || submitting}
                        />
                     )}
                  </div>
               </React.Fragment>
            );
         })}
      </div>
   );
});

ProjectListAutomationTumblers.displayName = 'ProjectListAutomationTumblers';
