import { classNames, IListOption, LazyDropdown } from '@yandex-infracloud-ui/libs';
import { Button, Icon, Menu } from 'lego-on-react';
import { default as React, useCallback, useEffect, useState } from 'react';
import { AutomationType } from '../../../models';
import { useConstants } from '../../../shared';

import addSvg from './add.svg';
import classes from './AddCreditButton.module.css';

interface Props {
   alreadyUsed: string[];
   className?: string;
   type: AutomationType;

   onSelect(field: string): void;
}

export const AddCreditButton = React.memo(({ type, onSelect, alreadyUsed, className }: Props) => {
   // region hooks
   const { constants } = useConstants();
   const [opened, setOpened] = useState(false);
   const [allLimits, setAllLimits] = useState<string[]>([]);
   const [options, setOptions] = useState<IListOption[]>([]);
   // endregion

   // region effects
   // Установка вариантов лимитов из констант (после загрузки констант)
   useEffect(() => {
      const v = constants
         ? constants.project_automation_limits.filter(limit =>
              type === 'dns' ? limit === 'max_dns_fixes' : limit !== 'max_dns_fixes',
           )
         : [];

      setAllLimits(v);
   }, [constants, type]);

   useEffect(() => {
      const o = allLimits.filter(l => !alreadyUsed.includes(l)).map(l => ({ value: l, name: l } as IListOption));

      setOptions(o);
   }, [allLimits, alreadyUsed]);
   // endregion

   // region handlers
   const onChangeOpened = useCallback((v: boolean) => setOpened(v), []);

   const close = useCallback(() => setOpened(false), []);

   const submit = useCallback(
      (fields: string[]) => {
         if (fields && fields.length > 0) {
            onSelect(fields[0]);
            close();
         }
      },
      [close, onSelect],
   );

   // endregion

   // region render
   if (options.length === 0) {
      return null;
   }

   const switcher = (
      <Button theme={'link'} size={'s'} cls={classNames(classes.addButton, className)} icon={<Icon url={addSvg} />}>
         Add credit
      </Button>
   );

   return (
      <LazyDropdown switcher={switcher} opened={opened} onChangeOpened={onChangeOpened}>
         <Menu theme={'normal'} size={'s'} type={'radio'} onChange={submit as any}>
            {options.map(o => (
               <Menu.Item key={o.value} val={o.value}>
                  {o.name}
               </Menu.Item>
            ))}
         </Menu>
      </LazyDropdown>
   );
   // endregion
});

AddCreditButton.displayName = 'AddCreditButton';
