import { UpdateTimerMode } from '@yandex-infracloud-ui/libs-next';

import { IScenario } from '../../../models';
import { OperationStatusType } from '../components/OperationStatus/OperationStatus';

export enum ActionType {
   BeforeLoading,
   Loaded,
   SetLoading,
}

interface IActionBeforeLoading {
   type: ActionType.BeforeLoading;
}

interface IActionSetLoading {
   isLoading: boolean;
   type: ActionType.SetLoading;
}

interface IActionLoadedScenario {
   scenario: IScenario;
   type: ActionType.Loaded;
}

type Action = IActionBeforeLoading | IActionLoadedScenario | IActionSetLoading;

export const initialState = {
   all: 0,
   done: 0,
   isLoading: true,
   mode: UpdateTimerMode.Slow,
   scenario: null as IScenario | null,
};

type State = Readonly<typeof initialState>;

const finalStatuses = new Set([OperationStatusType.Finished, OperationStatusType.Canceled]);

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.BeforeLoading: {
         return {
            ...state,
            isLoading: true,
         };
      }

      case ActionType.Loaded: {
         const hosts = action.scenario.hosts ?? [];
         const done = hosts.reduce((acc, item) => {
            return item.status === OperationStatusType.Done ? acc + 1 : acc;
         }, 0);

         return {
            ...state,
            all: hosts.length,
            done,
            isLoading: false,
            mode: finalStatuses.has(action.scenario.status) ? UpdateTimerMode.Slow : UpdateTimerMode.Fast,
            scenario: { ...action.scenario, workStatus: action.scenario.labels.WORK_STATUS },
         };
      }

      case ActionType.SetLoading: {
         return {
            ...state,
            isLoading: action.isLoading,
         };
      }

      default:
         return state;
   }
};
