import {
   EMPTY_VALUE,
   formatDate,
   isEmpty,
   IUserOrGroup,
   Loader,
   Rows,
   useDismounted,
   UserName,
   UserOrGroupType,
   useUpdateTimer,
} from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';

import * as React from 'react';
import { useCallback, useEffect, useMemo, useReducer, useRef } from 'react';
import { Route, RouteComponentProps, StaticContext, Switch } from 'react-router';
import { Subscription } from 'rxjs';
import { finalize, takeUntil } from 'rxjs/operators';

import { ROUTE_LINKS } from '../../../models';
import { scenariosApi } from '../../../services';
import {
   Field,
   HeaderWithBackLink,
   IScenarioContextValue,
   ITab,
   Page,
   ScenarioContext,
   Tabs,
   TrackerLink,
} from '../../../shared';
import { OperationStatus } from '../components/OperationStatus/OperationStatus';
import { ScenarioActionButtons } from '../components/ScenarioActionButtons/ScenarioActionButtons';
import { ScenarioErrorMessages } from '../components/ScenarioErrorMessages/ScenarioErrorMessages';
import { IScenarioMaintenanceTimeArgs } from '../models';
import { SCENARIO_DEFAULT_TABS } from '../tabs';
import styles from './Scenario.module.css';
import { ActionType, initialState, reducer } from './Scenario.state';

interface IProps {
   id: string;
}

export const Scenario = React.memo((props: RouteComponentProps<IProps, StaticContext, { prev: any }>) => {
   const { id } = props.match.params;

   const routes = SCENARIO_DEFAULT_TABS.map(r => <Route key={r.path as string} {...r} />);

   // region hooks
   const dismounted = useDismounted();
   const [{ done, isLoading, mode, scenario }, dispatch] = useReducer(reducer, initialState);
   const previousRequest = useRef<Subscription>();
   // endregion

   const maintenanceArgs = scenario?.script_args as IScenarioMaintenanceTimeArgs;
   const maintenanceStartTime = maintenanceArgs?.maintenance_start_time;
   const maintenanceEndTime = maintenanceArgs?.maintenance_end_time;

   // region handlers
   const cancel = useCallback(() => props.history.push(ROUTE_LINKS.scenarios()), [props.history]);
   // endregion

   // region helpers
   const updateItem = useCallback(
      (silent = false) => {
         if (!silent) {
            dispatch({ type: ActionType.BeforeLoading });
         }
         if (previousRequest.current) {
            previousRequest.current.unsubscribe();
         }

         previousRequest.current = scenariosApi
            .getById(id)
            .pipe(
               takeUntil(dismounted),
               finalize(() => (previousRequest.current = undefined)),
            )
            .subscribe(
               item => dispatch({ type: ActionType.Loaded, scenario: item }),
               resp => {
                  dispatch({ type: ActionType.SetLoading, isLoading: false });
                  toasts.apiError('Scenario loading', resp);

                  const backUrl =
                     props.location.state && props.location.state.prev
                        ? props.location.state.prev
                        : ROUTE_LINKS.scenarios();

                  props.history.push(backUrl);
               },
            );
      },
      [dismounted, id, props.history, props.location.state],
   );

   const updateItemSilent = useCallback(() => updateItem(true), [updateItem]);
   // endregion

   // region effects
   useEffect(updateItem, [dismounted, id]); // eslint-disable-line react-hooks/exhaustive-deps

   useUpdateTimer({ fast: 15000, slow: 60000, mode, callback: updateItemSilent });

   const contextValue = useMemo(() => ({ scenario, isLoading } as IScenarioContextValue), [isLoading, scenario]);

   // region render
   if (isLoading || !scenario) {
      return (
         <Page title={`Scenario ${id}`}>
            <Loader text={'Scenario is loading'} />
         </Page>
      );
   }

   const tabs = SCENARIO_DEFAULT_TABS.map(
      r =>
         ({
            header: r.name ? r.name.toUpperCase() : r.name,
            id: r.path,
            url: (r.path as string).replace(':id', id),
         } as ITab),
   );

   const issuerRecord: IUserOrGroup = {
      type: UserOrGroupType.People,
      id: scenario?.issuer.replace(/^@/, '').replace(/@$/, ''),
   };

   const header = (
      <>
         <HeaderWithBackLink text={scenario.name} url={ROUTE_LINKS.scenarios()} location={props.location}>
            <div className={styles.actions}>
               <ScenarioActionButtons item={scenario} onCancel={cancel} />
            </div>
         </HeaderWithBackLink>

         <div className={styles.details}>
            <div className={styles.column}>
               <Field title={'Status'}>
                  <OperationStatus status={scenario.status} />
               </Field>
               <Field title={'Work status'}>
                  <OperationStatus status={scenario.workStatus} />
               </Field>

               <Field title={'Progress'}>
                  {scenario.scenario_type}: {done} / {scenario.hosts ? scenario.hosts.length : 0}
               </Field>
            </div>

            <div className={styles.column}>
               <Field title={'Ticket'}>
                  {scenario.ticket_key ? <TrackerLink ticket={scenario.ticket_key} /> : EMPTY_VALUE}
               </Field>

               <Field title={'Issuer'}>
                  <UserName value={issuerRecord} avatar={true} />
               </Field>

               {isEmpty(scenario.errors) ? null : (
                  <Field title={'Errors'}>
                     <ScenarioErrorMessages errors={scenario.errors} />
                  </Field>
               )}

               {isEmpty(scenario.message) || scenario.message === 'no message' ? null : (
                  <Field title={'Message'}>
                     <Rows text={scenario.message} />
                  </Field>
               )}
            </div>
            <div className={styles.column}>
               <Field title={'Maintenance Start'}>
                  {maintenanceStartTime ? formatDate(new Date(maintenanceStartTime * 1000)) : EMPTY_VALUE}
               </Field>
               <Field title={'Maintenance End'}>
                  {maintenanceEndTime ? formatDate(new Date(maintenanceEndTime * 1000)) : EMPTY_VALUE}
               </Field>
            </div>
         </div>

         <Tabs items={tabs} prev={props.location.state ? props.location.state.prev : null} />
      </>
   );

   return (
      <ScenarioContext.Provider value={contextValue}>
         <Page header={header} title={scenario.name ? `Scenario ${scenario.name}` : `Scenario ${id}`}>
            <Switch>{routes}</Switch>
         </Page>
      </ScenarioContext.Provider>
   );
   // endregion
});

Scenario.displayName = 'Scenario';
