import { Card } from '@yandex-data-ui/common';
import { isEmpty, Loader, UpdateTimerMode, useDismounted, useUpdateTimer } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { useCallback, useContext, useEffect, useRef, useState } from 'react';
import { useTitle } from 'react-use';
import { Subscription } from 'rxjs';
import { takeUntil } from 'rxjs/operators';
import { scenariosApi } from '../../../../services';
import { ScenarioContext } from '../../../../shared';
import { ScenarioStagesTable } from '../state/ScenarioStagesTable';

export const ScenarioSnapshot = React.memo(() => {
   const [isLoading, setLoading] = useState<boolean>(true);
   const context = useContext(ScenarioContext);
   const scenario = context.scenario;
   const scenarioId = context.scenario?.scenario_id!;

   useTitle(`Scenario ${scenario?.name ?? ''} state`);

   const [scenarioState, setScenarioState] = useState();
   const dismounted = useDismounted();
   const previousRequest = useRef<Subscription>();

   const updateItem = useCallback(() => {
      if (previousRequest.current) {
         previousRequest.current.unsubscribe();
      }

      scenariosApi
         .getStateById(String(scenarioId))
         .pipe(takeUntil(dismounted))
         .subscribe(item => {
            setScenarioState(item);
            setLoading(false);
         });
   }, [dismounted, scenarioId, setScenarioState]);

   const updateItemSilent = useCallback(() => updateItem(), [updateItem]);
   // endregion

   // region effects
   useEffect(updateItem, [dismounted, scenario]); // eslint-disable-line react-hooks/exhaustive-deps

   useUpdateTimer({ fast: 15000, slow: 60000, mode: UpdateTimerMode.Slow, callback: updateItemSilent });

   if (isLoading) {
      return <Loader text={'Scenario state is loading'} />;
   }

   return isEmpty(scenarioState) ? (
      <span>Snapshot is empty.</span>
   ) : (
      <>
         <div style={{ display: 'flex' }}>
            <Card theme={'warning'}>
               <div style={{ padding: 10, maxWidth: 650 }}>
                  <strong>Early access</strong>
                  <br />
                  Be aware that alpha versions may be less stable than final version officially released to the public.
                  You may encounter problems such as app crashes ro features that do not work properly.
               </div>
            </Card>
         </div>
         <ScenarioStagesTable scenarioState={scenarioState} stages={(scenarioState as any).stages} />
      </>
   );
});

ScenarioSnapshot.displayName = 'ScenarioSnapshot';
