import { Progress } from '@yandex-data-ui/common';
import { classNames } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { useContext, useState } from 'react';
import { ScenarioStageStatus } from '../../../../models';
import { ClickOnlyUnmovedTrap, ScenarioContext } from '../../../../shared';
import { OPERATION_STATUS_ICONS } from '../../components/OperationStatus/OperationStatus';
import styles from './ScenarioStages.module.css';
import { Sidebar } from './Sidebar';
import { StageHostTableItem } from './StageHostTableItem';
import { StageHostInfoItem } from './StageHostInfoItem';

interface IItemProps {
   description?: any;
   level: number;
   name: string;
   stage: any;
   status: ScenarioStageStatus | 'group';
   groupNumberOfHosts?: number | undefined;
}

export const ScenarioStagesTableItem = React.memo(({ stage, status, name, level, groupNumberOfHosts }: IItemProps) => {
   const { scenario } = useContext(ScenarioContext);
   const [isSidebarVisible, setSidebarVisible] = useState(false);
   const stageHosts = Object.keys(stage.hosts || {}).map(key => stage.hosts[key]);
   const finishedHosts = stageHosts.filter(item => item.status === 'finished').length;
   const processingHosts = stageHosts.filter(item => item.status !== 'finished').length;
   const hostCount = groupNumberOfHosts || scenario?.hosts?.length!;

   const toggleSidebarVisible = () => {
      setSidebarVisible(s => !s);
   };

   // render
   return (
      <>
         <ClickOnlyUnmovedTrap
            as={'tr'}
            onClick={toggleSidebarVisible}
            data-e2e={'HostRow:first'}
            className={classNames(styles.row, {
               [styles.selected]: isSidebarVisible,
            })}
         >
            <td className={styles.nowrap}>
               <div style={{ paddingLeft: level * 12 }}>
                  {OPERATION_STATUS_ICONS.get(status as any)} {name}
               </div>
            </td>
            <td className={styles.nowrap}>
               {(processingHosts > 0 || (finishedHosts > 0 && finishedHosts < hostCount)) && (
                  <div style={{ display: 'flex', alignItems: 'center' }}>
                     <div style={{ width: '80px' }}>
                        <Progress
                           view={'thin'}
                           stack={[
                              {
                                 value: (finishedHosts * 100) / hostCount,
                                 color: 'var(--yc-color-infographics-positive-medium)',
                              },
                           ]}
                        />
                     </div>
                     <span style={{ color: 'var(--muted-text)', paddingLeft: '8px' }}>
                        {' '}
                        <span style={{ color: 'var(--yc-color-infographics-positive-medium)' }}>
                           {finishedHosts}
                        </span> /{' '}
                        <span style={{ color: 'var(--yc-color-infographics-info-medium)' }}>{processingHosts}</span> /{' '}
                        {hostCount}
                     </span>
                  </div>
               )}
            </td>
            <td
               className={classNames(styles.nowrap, styles.description, styles.textSecondary)}
               title={stage.description}
            >
               {stage.description}
            </td>
         </ClickOnlyUnmovedTrap>
         {isSidebarVisible && (
            <Sidebar onClose={() => setSidebarVisible(false)}>
               <div style={{ width: 600 }}>
                  <h2 style={{ padding: 20 }}>
                     {OPERATION_STATUS_ICONS.get(status as any)} {name}
                  </h2>

                  <StageHostInfoItem
                     description={stage.description}
                     status={stage.status}
                     msg={stage.msg}
                     stage={stage}
                  />

                  {stage.hasOwnProperty('hosts') && Object.entries(stage.hosts).length > 0 && (
                     <table style={{ width: '100%' }}>
                        <tbody>
                           {Object.entries(stage.hosts).map(([hostName, item]: [string, any], index: any) => {
                              return (
                                 <StageHostTableItem
                                    key={item.uid + hostName}
                                    name={hostName}
                                    status={item.status as any}
                                    msg={item.msg}
                                    level={0}
                                 />
                              );
                           })}
                        </tbody>
                     </table>
                  )}
               </div>
            </Sidebar>
         )}

         {stage.data?.hasOwnProperty('group_stages_containers')
            ? stage.data.group_stages_containers.map((item: any, index: any) => {
                 return (
                    <ScenarioStagesTableItem
                       key={stage.uid + item.group_id}
                       name={`GROUP ID: ${item.group_id} ${item.group_name || ''}`}
                       status={'group'}
                       stage={item}
                       level={level + 2}
                       groupNumberOfHosts={item.number_of_hosts}
                    />
                 );
              })
            : stage.stages.map((item: any, index: any) => {
                 return (
                    <ScenarioStagesTableItem
                       key={item.uid}
                       name={item.name}
                       status={item.status as any}
                       stage={item}
                       level={level + 1}
                       groupNumberOfHosts={groupNumberOfHosts}
                    />
                 );
              })}
      </>
   );
});

ScenarioStagesTableItem.displayName = 'ScenarioStagesTableItem';
