import {
   classNames,
   EMPTY_VALUE,
   formatDate,
   formatNumber,
   IUserOrGroup,
   plural,
   styleHelpers,
   UserName,
   UserOrGroupType,
} from '@yandex-infracloud-ui/libs-next';
import { WarnIcon } from 'design/icons';
import * as H from 'history';
import * as React from 'react';
import { Children, SyntheticEvent, useCallback, useState } from 'react';
import { Link } from 'react-router-dom';
import { IScenario, ROUTE_LINKS } from '../../../models';

import { ClickOnlyUnmovedTrap, TrackerLink } from '../../../shared';
import { OperationStatus } from '../components/OperationStatus/OperationStatus';
import { ScenarioJsonDetails } from '../components/ScenarioJsonDetails/ScenarioJsonDetails';
import { IScenarioMaintenanceTimeArgs } from '../models';

import classes from './ScenarioItem.module.css';

interface IProps {
   item: IScenario;
   location?: H.Location;

   onCanceled(id: number): void;
}

export const ScenarioItem = React.memo(({ item, location, onCanceled }: IProps) => {
   const { scenario_id, name, issuer } = item;
   const issuerRecord: IUserOrGroup = {
      type: UserOrGroupType.People,
      id: issuer.replace(/^@/, '').replace(/@$/, ''),
   };

   const maintenanceArgs = item.script_args as IScenarioMaintenanceTimeArgs;
   const maintenanceStartTime = maintenanceArgs?.maintenance_start_time;
   const maintenanceEndTime = maintenanceArgs?.maintenance_end_time;

   // hooks
   const [expanded, setExpanded] = useState(false);

   // handlers

   const toggleExpand = useCallback((e: SyntheticEvent) => {
      const target = e.target as EventTarget & Element;

      if (target.tagName === 'TD') {
         setExpanded(v => !v);
      }
   }, []);

   // render
   const errorCount = Object.keys(item?.errors ?? {}).length;

   const firstRow = (
      <ClickOnlyUnmovedTrap
         as={'tr'}
         onClick={toggleExpand}
         data-e2e={'ScenarioItem:first'}
         className={classNames(classes.row, {
            [classes.expanded]: expanded,
         })}
      >
         <td className={classes.name}>
            <Link
               to={{
                  pathname: ROUTE_LINKS.scenario(scenario_id),
                  state: { prev: location },
               }}
            >
               {name}

               {errorCount > 0 ? (
                  <span
                     title={`Has ${formatNumber(errorCount)} ${plural(errorCount, 'error', 'errors')}`}
                     className={styleHelpers.leftSpace}
                  >
                     <WarnIcon color={'var(--error-color)'} fixedWidth={true} />
                  </span>
               ) : null}
            </Link>
         </td>

         <td className={classes.status}>
            <OperationStatus status={item.labels.WORK_STATUS} />
         </td>

         <td className={classes.type}>{item.scenario_type}</td>

         <td className={classes.maintenanceTime}>
            {maintenanceStartTime ? formatDate(new Date(maintenanceStartTime * 1000)) : EMPTY_VALUE}
         </td>

         <td className={classes.maintenanceTime}>
            {maintenanceEndTime ? formatDate(new Date(maintenanceEndTime * 1000)) : EMPTY_VALUE}
         </td>

         <td className={classes.issuer}>
            <UserName value={issuerRecord} avatar={true} />
         </td>

         <td className={classes.ticket}>{item.ticket_key ? <TrackerLink ticket={item.ticket_key} /> : EMPTY_VALUE}</td>
      </ClickOnlyUnmovedTrap>
   );

   return (
      <>
         {firstRow}

         {expanded ? (
            <tr data-e2e={'ScenarioItem:second'} className={classes.additional}>
               <td colSpan={Children.count(firstRow.props.children)}>
                  <ScenarioJsonDetails short={true} value={item} />
               </td>
            </tr>
         ) : null}
      </>
   );
});

ScenarioItem.displayName = 'ScenarioItem';
