import { isEqual, PER_PAGE_DEFAULT, UpdateTimerMode } from '@yandex-infracloud-ui/libs-next';

import { IListResult, IScenario } from '../../../models';
import { OperationStatusType } from '../components/OperationStatus/OperationStatus';
import { IssuerFilterType } from '../models';

export enum ActionType {
   BeforeLoading,
   Loaded,
   SetLoading,
   SetFilterQuery,
   SetOwner,
   SetOperationStatus,
   SetScenarioType,
   SetPage,
   SetPerPage,
}

interface IActionBeforeLoaded {
   type: ActionType.BeforeLoading;
}

interface IActionLoaded {
   result: IListResult<IScenario>;
   type: ActionType.Loaded;
}

interface IActionSetLoading {
   isLoading: boolean;
   type: ActionType.SetLoading;
}

interface IActionSetFilterQuery {
   query: string | undefined;
   type: ActionType.SetFilterQuery;
}

interface IActionSetOwner {
   issuerType: IssuerFilterType;
   type: ActionType.SetOwner;
}

interface IActionSetOperationStatus {
   operationStatus: Set<string>;
   type: ActionType.SetOperationStatus;
}

interface IActionSetScenarioType {
   scenarioType: Set<string>;
   type: ActionType.SetScenarioType;
}

interface IActionSetPage {
   page: number;
   type: ActionType.SetPage;
}

interface IActionSetPerPage {
   perPage: number;
   type: ActionType.SetPerPage;
}

export const excludedOperationStatuses = new Set([
   OperationStatusType.Rejected,
   OperationStatusType.Canceled,
   OperationStatusType.Finished,
]) as Set<string>;

export const initialState = {
   isLoading: true,
   issuerType: IssuerFilterType.My,
   operationStatus: new Set([]) as Set<string>,
   scenarioType: new Set([]) as Set<string>,
   items: [] as IScenario[],
   mode: UpdateTimerMode.Slow,
   page: 1,
   perPage: PER_PAGE_DEFAULT,
   query: '',
   total: 0,
};

type Action =
   | IActionBeforeLoaded
   | IActionLoaded
   | IActionSetLoading
   | IActionSetFilterQuery
   | IActionSetOwner
   | IActionSetOperationStatus
   | IActionSetScenarioType
   | IActionSetPage
   | IActionSetPerPage;

export type State = Readonly<typeof initialState>;

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.BeforeLoading: {
         return {
            ...state,
            isLoading: true,
         };
      }

      case ActionType.Loaded: {
         const newItems = action.result.result;

         const pendingOperation = newItems.some(a => a.status !== OperationStatusType.Finished);

         const items = newItems.reduce((acc, updatedOperation) => {
            const operation = state.items.find(op => op.scenario_id === updatedOperation.scenario_id);
            if (operation && isEqual(updatedOperation, operation)) {
               acc.push(operation);
            } else {
               acc.push(updatedOperation);
            }

            return acc;
         }, [] as IScenario[]);

         return {
            ...state,
            isLoading: false,
            items: state.items.length === 0 ? newItems : items,
            mode: pendingOperation ? UpdateTimerMode.Fast : UpdateTimerMode.Slow,
            total: action.result.total || state.items.length,
         };
      }

      case ActionType.SetLoading: {
         return {
            ...state,
            isLoading: action.isLoading,
         };
      }

      case ActionType.SetFilterQuery: {
         return {
            ...state,
            query: action.query || '',
         };
      }

      case ActionType.SetOwner: {
         return {
            ...state,
            issuerType: action.issuerType,
         };
      }

      case ActionType.SetOperationStatus: {
         return {
            ...state,
            operationStatus: action.operationStatus,
         };
      }

      case ActionType.SetScenarioType: {
         return {
            ...state,
            scenarioType: action.scenarioType,
         };
      }

      case ActionType.SetPage: {
         return {
            ...state,
            page: action.page,
         };
      }

      case ActionType.SetPerPage: {
         return {
            ...state,
            perPage: action.perPage,
         };
      }

      default:
         return state;
   }
};
