import {
   fromQuery,
   Loader,
   PER_PAGE_DEFAULT,
   queryValueToSet,
   setToQueryValue,
   useBehaviourSubject,
   useDebouncedValue,
   useDismounted,
   useUpdateTimer,
} from '@yandex-infracloud-ui/libs-next';
import { Pagination, useUrlUpdater, toasts } from '@yandex-infracloud-ui/libs';

import { Button } from '@yandex-data-ui/common';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useReducer, useRef } from 'react';
import { RouteComponentProps } from 'react-router';
import { Subscription } from 'rxjs';
import { finalize, takeUntil } from 'rxjs/operators';

import { globalState, IConstants, IListResult, IScenario, IScenarioFilters } from '../../../models';
import { scenariosApi } from '../../../services';
import { Page } from '../../../shared';
import { useOpenModal } from '../components/config';
import { CreateScenarioModal } from '../components/CreateDefaultModal/CreateScenarioModal';
import { ScenarioFilters } from '../components/ScenarioFilters/ScenarioFilters';
import { IssuerFilterType } from '../models';
import { ScenarioItem } from './ScenarioItem';

import styles from './ScenarioList.module.css';
import { ActionType, initialState, reducer, excludedOperationStatuses } from './ScenarioList.state';

export interface IUrlParams {
   issuer?: IssuerFilterType;
   page?: string;
   perPage?: string;
   q?: string;
   work_status?: string;
   type?: string;
}

interface Props extends RouteComponentProps<IUrlParams> {
   constants: IConstants | null;
}

export const ScenarioList = React.memo((props: Props) => {
   const urlParams: IUrlParams = fromQuery(props.location.search);

   // hooks

   const defaultOperationStatuses = new Set(
      props.constants?.scenario_work_statuses.filter(e => !excludedOperationStatuses.has(e)),
   );

   const dismounted = useDismounted();

   const previousRequest = useRef<Subscription>();

   const { openModal } = useOpenModal();

   const [
      { items, total, isLoading, query, issuerType, operationStatus, scenarioType, mode, page, perPage },
      dispatch,
   ] = useReducer(reducer, {
      ...initialState,
      issuerType: urlParams.issuer || IssuerFilterType.My,
      operationStatus: urlParams.work_status
         ? urlParams.work_status === 'all'
            ? initialState.operationStatus
            : queryValueToSet(urlParams.work_status)
         : defaultOperationStatuses,
      scenarioType: urlParams.type ? queryValueToSet(urlParams.type) : initialState.scenarioType,
      page: urlParams.page ? parseInt(urlParams.page, 10) : 1,
      perPage: urlParams.perPage ? parseInt(urlParams.perPage, 10) : PER_PAGE_DEFAULT,
      query: urlParams.q || '',
   });

   const debouncedQuery = useDebouncedValue(query, 500);

   const user = useBehaviourSubject(globalState.user);

   // helpers
   const updateList = useCallback(
      (silent = false) => {
         if (previousRequest.current) {
            previousRequest.current.unsubscribe();
         }

         if (!silent) {
            dispatch({ type: ActionType.BeforeLoading });
         }

         previousRequest.current = scenariosApi
            .getList(perPage, (page - 1) * perPage, {
               issuer: issuerType === IssuerFilterType.My ? (user ? user.login : undefined) : undefined,
               workStatus: operationStatus.size === 0 ? undefined : operationStatus,
               type: scenarioType,
               query: debouncedQuery,
            })
            .pipe(
               finalize(() => (previousRequest.current = undefined)),
               takeUntil(dismounted),
            )
            .subscribe(
               (resp: IListResult<IScenario>) => {
                  if (resp.result.length === 0 && page > 1) {
                     dispatch({ type: ActionType.SetPage, page: 1 });

                     return;
                  }

                  dispatch({ type: ActionType.Loaded, result: resp });
               },
               (resp: any) => {
                  if (!silent) {
                     dispatch({ type: ActionType.SetLoading, isLoading: false });
                  }
                  toasts.apiError('Scenarios loading', resp);
               },
            );
      },
      [perPage, page, issuerType, user, operationStatus, scenarioType, debouncedQuery, dismounted],
   );

   const updateListSilent = useCallback(() => updateList(true), [updateList]);

   // handlers
   const updateQuery = useCallback(
      (q: string) =>
         dispatch({
            query: q,
            type: ActionType.SetFilterQuery,
         }),
      [],
   );

   const updateIssuer = useCallback(
      (v: string) =>
         dispatch({
            issuerType: v as IssuerFilterType,
            type: ActionType.SetOwner, // TODO: change SetOwner -> SetIssuer
         }),
      [],
   );

   const updateStatus = useCallback((v: Set<string>) => {
      dispatch({
         operationStatus: v,
         type: ActionType.SetOperationStatus,
      });
   }, []);

   const updateType = useCallback(
      (v: Set<string>) =>
         dispatch({
            scenarioType: v,
            type: ActionType.SetScenarioType,
         }),
      [],
   );

   const updateFilters = useCallback(
      (f: Required<IScenarioFilters>) => {
         updateIssuer(f.issuer);
         updateQuery(f.query);
         updateStatus(f.workStatus);
         updateType(f.type);
      },
      [updateIssuer, updateQuery, updateStatus, updateType],
   );

   const openCreateModal = useCallback(() => {
      openModal(CreateScenarioModal, {
         history: props.history,
      });
   }, [openModal, props.history]);

   const onCanceled = useCallback(() => updateList(), [updateList]);

   const onPageToggle = useCallback(
      (e: SyntheticEvent, newPage: number) => {
         if (newPage !== page) {
            dispatch({ type: ActionType.SetPage, page: newPage });
         }
      },
      [page],
   );

   const onPerPageToggle = useCallback(
      (e: SyntheticEvent, newPerPage: number) => {
         if (newPerPage !== perPage) {
            dispatch({ type: ActionType.SetPerPage, perPage: newPerPage });
         }
      },
      [perPage],
   );

   // effects
   useUpdateTimer({ fast: 15000, slow: 60000, mode, callback: updateListSilent });

   useEffect(updateList, [perPage, page, issuerType, operationStatus, scenarioType, debouncedQuery, dismounted, user]); // eslint-disable-line react-hooks/exhaustive-deps

   // TODO fix in ui-components: do not update twice
   useUrlUpdater(
      props,
      () => ({
         issuer: issuerType === IssuerFilterType.My ? undefined : issuerType,
         work_status: operationStatus.size === 0 ? 'all' : setToQueryValue(operationStatus),
         type: setToQueryValue(scenarioType),
         page: page > 1 ? page.toString() : undefined,
         perPage: perPage !== PER_PAGE_DEFAULT ? perPage.toString() : undefined,
         q: query,
      }),
      [query, issuerType, operationStatus, scenarioType, page, perPage],
   );

   // render

   return (
      <Page title={'Scenarios'}>
         <div className={styles.topBar}>
            <ScenarioFilters
               filters={{
                  issuer: issuerType,
                  query,
                  workStatus: operationStatus,
                  type: scenarioType,
               }}
               onChange={updateFilters}
            />
            <Button view={'action'} size={'m'} onClick={openCreateModal}>
               Create
            </Button>
         </div>

         <table className={styles.table}>
            <thead>
               <tr>
                  <th className={styles.name}>Name</th>
                  <th className={styles.status}>Status</th>
                  <th className={styles.type}>Type</th>
                  <th className={styles.maintenanceTime}>Maintenance Start</th>
                  <th className={styles.maintenanceTime}>Maintenance End</th>
                  <th className={styles.issuer}>Issuer</th>
                  <th className={styles.ticket}>Ticket</th>
                  <th />
               </tr>
            </thead>

            <tbody>
               {isLoading
                  ? null
                  : items.map(item => (
                       <ScenarioItem
                          item={item}
                          key={item.scenario_id}
                          location={props.location}
                          onCanceled={onCanceled}
                       />
                    ))}
            </tbody>
         </table>

         {items.length === 0 && !isLoading ? <div className={styles.noData}>No data</div> : null}

         <Loader text={'Scenarios loading'} visible={isLoading} cls={styles.loader} />

         <Pagination
            perPage={perPage}
            perPageVariants={[25, 50, 100]}
            total={Math.ceil(total / perPage)}
            value={page}
            onChange={onPageToggle}
            onPerPageChange={onPerPageToggle}
         />
      </Page>
   );
});

ScenarioList.displayName = 'ScenarioList';
