import { IModalProps, ModalLayout, toasts } from '@yandex-infracloud-ui/libs';
import { clearParams, DISMISS_REASON, useDismounted } from '@yandex-infracloud-ui/libs-next';
import React, { useCallback, useMemo, useState } from 'react';
import { useHistory } from 'react-router-dom';
import { takeUntil } from 'rxjs/operators';
import { scenariosApi } from 'services';

import { ROUTE_LINKS, ScenarioParams } from '../../../../models';
import { ScenarioType } from '../../models';
import { CreateScenarioHostsTransferForm, scenarioHostsTransferFields } from './CreateScenarioHostsTransferForm';
import {
   CreateScenarioReservedHostsTransferForm,
   scenarioReservedHostsTransferFields,
} from './CreateScenarioReservedHostsTransferForm';

import { CreateScenarioItdcMaintenanceForm, scenarioItdcMaintenanceFields } from './CreateScenarioItdcMaintenanceForm';

import { CreateScenarioNOCHardForm, scenarioNOCHardFields } from './CreateScenarioNOCHardForm';

import { CreateScenarioNOCSoftForm, scenarioNOCSoftFields } from './CreateScenarioNOCSoftForm';

interface ScenarioModalProps {
   params: Partial<ScenarioParams>;
}

interface Props extends IModalProps<ScenarioModalProps> {
   initialValue?: Partial<ScenarioParams>;
}

export const CreateScenarioModal = React.memo(({ initialValue, ok, cancel }: Props) => {
   // hooks
   const history = useHistory();
   const dismounted = useDismounted();
   const [type, setType] = useState(ScenarioType.HostsTransfer);

   // handlers
   const handleScenarioTypeChange = useCallback(
      (value: string) => {
         setType(value as ScenarioType.HostsTransfer);
      },
      [setType],
   );

   const handleSubmit = useCallback(
      (v, formikHelpers) => {
         let request;
         let params: Partial<ScenarioParams>;

         switch (type) {
            case ScenarioType.HostsTransfer: {
               params = clearParams(scenarioHostsTransferFields as any, v);
               params.script_args = v.script_args;
               request = scenariosApi.createHostsTransfer(params);

               break;
            }

            case ScenarioType.ReservedHostsTransfer: {
               params = clearParams(scenarioReservedHostsTransferFields as any, v);
               params.script_args = v.script_args;
               request = scenariosApi.createReservedHostsTransfer(params);

               break;
            }

            case ScenarioType.ITDCMaintenance: {
               params = clearParams(scenarioItdcMaintenanceFields as any, v);
               params.script_args = v.script_args;
               request = scenariosApi.createItdcMaintenance(params);

               break;
            }

            case ScenarioType.NOCHard: {
               params = clearParams(scenarioNOCHardFields as any, v);
               params.script_args = v.script_args;
               request = scenariosApi.createNocHard(params);

               break;
            }

            case ScenarioType.NOCSoft: {
               params = clearParams(scenarioNOCSoftFields as any, v);
               params.script_args = v.script_args;
               request = scenariosApi.createNocSoft(params);

               break;
            }
         }

         request.pipe(takeUntil(dismounted)).subscribe(
            resp => {
               ok({ params });
               toasts.success(`Scenario ${params.name} has been created`);
               history?.push(ROUTE_LINKS.scenario(resp.scenario_id));
            },
            e => {
               formikHelpers.setSubmitting(false);
               toasts.apiError('Scenario creating', e);
            },
         );
      },
      [type, dismounted, ok, history],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const Form = useMemo(() => {
      switch (type) {
         case ScenarioType.HostsTransfer:
            return CreateScenarioHostsTransferForm;
         case ScenarioType.ReservedHostsTransfer:
            return CreateScenarioReservedHostsTransferForm;
         case ScenarioType.ITDCMaintenance:
            return CreateScenarioItdcMaintenanceForm;
         case ScenarioType.NOCHard:
            return CreateScenarioNOCHardForm;
         case ScenarioType.NOCSoft:
            return CreateScenarioNOCSoftForm;
         default:
            throw new Error('Unexpected scenario type');
      }
   }, [type]);

   return (
      <ModalLayout title={'New scenario'} showFooter={false} onDismiss={dismiss}>
         <Form
            initialValue={initialValue}
            onSubmit={handleSubmit}
            onDismiss={dismiss}
            onScenarioTypeChange={handleScenarioTypeChange}
         />
      </ModalLayout>
   );
});

CreateScenarioModal.displayName = 'CreateScenarioModal';
