import {
   CheckboxField2,
   DeclarativeFields,
   ExtendedFieldConfig,
   FormHooks,
   FormHooksContext,
   omitFields,
   SetField,
} from '@yandex-infracloud-ui/libs-next';
import { Form, Formik, FormikProps } from 'formik';
import { FormikHelpers } from 'formik/dist/types';
import { Button } from '@yandex-data-ui/common';
import React, { PropsWithChildren, useCallback, useMemo, useRef, useState } from 'react';
import classes from '../../../../design/commonModals.module.css';
import { ScenarioParams } from '../../../../models';
import { DevJson } from '../../../../shared';
import { TrackerIssueField } from '../../../form_fields2/TrackerIssueField';
import { DateTimeField } from '../fields/DateTimeField';
import { EnumField } from '../../../../rich_shared/form_fields2/EnumField';
import { nocHardScenarioSchema, ScenarioType } from '../../models';
import { ScenarioNameField } from '../fields/ScenarioNameField';
import { ScenarioTypeField } from '../fields/ScenarioTypeField';
import { getMaintenceTimeValue } from '../fields/formFields';

type ScenarioFormFields = ScenarioParams & {
   _scheduledMaintance: 'disabled' | 'start' | 'range';
   _resourceType: 'switch' | 'hosts';
};

export const scenarioNOCHardFields: ExtendedFieldConfig<ScenarioFormFields>[] = [
   {
      as: ScenarioTypeField,
      label: 'Scenario type',
      name: 'scenario_type',
      required: true,
   },
   {
      as: TrackerIssueField,
      label: 'Ticket',
      name: 'ticket_key',
      required: true,
   },
   {
      as: ScenarioNameField,
      label: 'Name',
      name: 'name',
      placeholder: 'Enter scenario name',
      required: true,
   },
   {
      as: EnumField,
      controlProps: {
         options: [
            { value: 'hosts', content: 'Hosts' },
            { value: 'switch', content: 'Switch' },
         ],
      },
      label: 'Resources',
      name: '_resourceType',
   },
   {
      as: SetField,
      label: 'Hosts',
      name: 'hosts',
      placeholder: 'Enter host name',
      required: true,
      hidden: (_, formikProps) => {
         return formikProps.values._resourceType !== 'hosts';
      },
   },
   {
      as: ScenarioNameField,
      label: 'Switch',
      name: 'script_args.switch' as any,
      placeholder: 'Enter switch',
      required: true,
      hidden: (_, formikProps) => {
         return formikProps.values._resourceType !== 'switch';
      },
   },
   {
      as: EnumField,
      label: 'Scheduled maintenance',
      controlProps: {
         options: [
            { value: 'range', content: 'Time range' },
            { value: 'start', content: 'Start time' },
            { value: 'disabled', content: 'Disabled' },
         ],
      },
      name: '_scheduledMaintance',
   },
   {
      as: DateTimeField,
      label: 'Maintenance start time',
      name: 'script_args.maintenance_start_time' as any,
      hidden: (_, formikProps) => {
         return formikProps.values._scheduledMaintance === 'disabled';
      },
   },
   {
      as: DateTimeField,
      label: 'Maintenance end time',
      name: 'script_args.maintenance_end_time' as any,
      hidden: (_, formikProps) => {
         return !(formikProps.values._scheduledMaintance === 'range');
      },
   },
   {
      as: CheckboxField2,
      label: 'Autostart',
      name: 'autostart',
   },
];

const initialValueBase: ScenarioFormFields = {
   name: '',
   autostart: true,
   hosts: new Set(),
   scenario_type: ScenarioType.NOCHard,
   script_args: {
      maintenance_start_time: getMaintenceTimeValue('maintenance_start_time', 'range'),
      maintenance_end_time: getMaintenceTimeValue('maintenance_end_time', 'range'),
   },
   _scheduledMaintance: 'range',
   _resourceType: 'hosts',
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <DeclarativeFields configs={scenarioNOCHardFields} />

      {children}
   </Form>
);

interface Props {
   initialValue?: Partial<ScenarioParams>;

   onSubmit(values: ScenarioParams, formikHelpers: FormikHelpers<ScenarioParams>): void;

   onScenarioTypeChange(type: string): void;

   onDismiss(): void;
}

export const CreateScenarioNOCHardForm: React.FC<Props> = React.memo(
   ({ onSubmit, onDismiss, onScenarioTypeChange, initialValue }) => {
      // hooks
      const [apiError, setApiError] = useState('');
      const formRef = useRef<FormikProps<ScenarioFormFields>>();

      const initialValuePatched = useMemo(() => ({ ...initialValueBase, ...initialValue }), [initialValue]);

      const onFormSubmit = useCallback(
         (values, formikHelpers) => onSubmit(omitFields(values, '_resourceType') as ScenarioParams, formikHelpers),
         [onSubmit],
      );

      const formHooks = useMemo(
         () =>
            ({
               onFieldChange: (name, value) => {
                  setApiError('');

                  if (!formRef.current) {
                     return;
                  }

                  if (name === 'scenario_type') {
                     if (typeof value !== 'string') {
                        throw new Error('Scenario type must be string');
                     }

                     onScenarioTypeChange(value);
                  }

                  if (name === '_scheduledMaintance') {
                     formRef.current.setFieldValue(
                        'script_args.maintenance_start_time',
                        getMaintenceTimeValue('maintenance_start_time', value),
                     );
                     formRef.current.setFieldValue(
                        'script_args.maintenance_end_time',
                        getMaintenceTimeValue('maintenance_end_time', value),
                     );
                  }

                  if (name === '_resourceType') {
                     if (value === 'hosts') {
                        formRef.current.setFieldValue('script_args.switch', undefined);
                     } else if (value === 'switch') {
                        formRef.current.setFieldValue('hosts', new Set());
                     }
                  }
               },
            } as FormHooks<ScenarioFormFields>),
         [onScenarioTypeChange],
      );

      // render
      return (
         <Formik initialValues={initialValuePatched} onSubmit={onFormSubmit} validationSchema={nocHardScenarioSchema}>
            {form => {
               formRef.current = form;

               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <InternalForm>
                        <DevJson>{{ form }}</DevJson>

                        <footer className={classes.footer}>
                           {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                           <Button
                              view={'clear'}
                              size={'m'}
                              className={classes.button}
                              onClick={onDismiss}
                              disabled={form.isSubmitting}
                           >
                              Cancel
                           </Button>

                           <Button
                              view={'action'}
                              size={'m'}
                              className={classes.button}
                              type={'submit'}
                              disabled={!form.isValid || form.isSubmitting || !form.dirty}
                           >
                              {form.values.autostart ? 'Create and start' : 'Create'}
                           </Button>
                        </footer>
                     </InternalForm>
                  </FormHooksContext.Provider>
               );
            }}
         </Formik>
      );
   },
);

CreateScenarioNOCHardForm.displayName = 'CreateScenarioNOCHardForm';
