import {
   DeclarativeFields,
   CheckboxField2,
   ExtendedFieldConfig,
   FormHooks,
   FormHooksContext,
   omitFields,
   InputField2,
   TextField2,
} from '@yandex-infracloud-ui/libs-next';
import { Form, Formik, FormikProps } from 'formik';
import { FormikHelpers } from 'formik/dist/types';
import { Button } from '@yandex-data-ui/common';
import React, { PropsWithChildren, useCallback, useMemo, useRef, useState } from 'react';
import classes from '../../../../design/commonModals.module.css';
import { ScenarioParams } from '../../../../models';
import { DevJson } from '../../../../shared';
import { TrackerIssueField } from '../../../form_fields2/TrackerIssueField';
import { nocSoftScenarioSchema, ScenarioType } from '../../models';
import { ScenarioNameField } from '../fields/ScenarioNameField';
import { ScenarioTypeField } from '../fields/ScenarioTypeField';

export const scenarioNOCSoftFields: ExtendedFieldConfig<ScenarioParams>[] = [
   {
      as: ScenarioTypeField,
      label: 'Scenario type',
      name: 'scenario_type',
      required: true,
   },
   {
      as: ScenarioNameField,
      label: 'Name',
      name: 'name',
      placeholder: 'Enter scenario name',
      required: true,
   },
   {
      as: TrackerIssueField,
      label: 'Ticket',
      name: 'ticket_key',
      required: true,
   },
   {
      as: InputField2,
      label: 'Switch',
      name: 'script_args.switch' as any,
      placeholder: 'Switch to collect hosts from',
      required: true,
   },
   {
      as: TextField2,
      hint: 'Will be logged',
      label: 'Comment',
      name: 'reason',
      placeholder: 'Enter a reason for project creation',
   },
   {
      as: CheckboxField2,
      label: 'Autostart',
      name: 'autostart',
   },
];

const initialValueBase: ScenarioParams = {
   name: '',
   autostart: true,
   scenario_type: ScenarioType.NOCSoft,
   script_args: {},
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <DeclarativeFields configs={scenarioNOCSoftFields} />

      {children}
   </Form>
);

interface Props {
   initialValue?: Partial<ScenarioParams>;

   onSubmit(values: ScenarioParams, formikHelpers: FormikHelpers<ScenarioParams>): void;

   onScenarioTypeChange(type: string): void;

   onDismiss(): void;
}

export const CreateScenarioNOCSoftForm: React.FC<Props> = React.memo(
   ({ onSubmit, onDismiss, initialValue, onScenarioTypeChange }) => {
      // hooks
      const [apiError, setApiError] = useState('');
      const formRef = useRef<FormikProps<ScenarioParams>>();

      const initialValuePatched = useMemo(() => ({ ...initialValueBase, ...initialValue }), [initialValue]);

      const onFormSubmit = useCallback(
         (values, formikHelpers) => onSubmit(omitFields(values) as ScenarioParams, formikHelpers),
         [onSubmit],
      );

      const formHooks = useMemo(
         () =>
            ({
               onFieldChange: (name, value) => {
                  setApiError('');

                  if (!formRef.current) {
                     return;
                  }

                  if (name === 'scenario_type') {
                     if (typeof value !== 'string') {
                        throw new Error('Scenario type must be string');
                     }

                     onScenarioTypeChange(value);
                  }
               },
            } as FormHooks<ScenarioParams>),
         [onScenarioTypeChange],
      );

      // render
      return (
         <Formik initialValues={initialValuePatched} onSubmit={onFormSubmit} validationSchema={nocSoftScenarioSchema}>
            {form => {
               formRef.current = form;

               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <InternalForm>
                        <DevJson>{{ form }}</DevJson>

                        <footer className={classes.footer}>
                           {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                           <Button
                              view={'clear'}
                              size={'m'}
                              className={classes.button}
                              onClick={onDismiss}
                              disabled={form.isSubmitting}
                           >
                              Cancel
                           </Button>

                           <Button
                              view={'action'}
                              size={'m'}
                              className={classes.button}
                              type={'submit'}
                              disabled={!form.isValid || form.isSubmitting || !form.dirty}
                           >
                              {form.values.autostart ? 'Create and start' : 'Create'}
                           </Button>
                        </footer>
                     </InternalForm>
                  </FormHooksContext.Provider>
               );
            }}
         </Formik>
      );
   },
);

CreateScenarioNOCSoftForm.displayName = 'CreateScenarioNOCSoftForm';
