import {
   CheckboxField2,
   DeclarativeFields,
   ExtendedFieldConfig,
   FormHooks,
   FormHooksContext,
   SetField,
   TextField2,
   InputField2,
} from '@yandex-infracloud-ui/libs-next';
import { Form, Formik, FormikHelpers, FormikProps } from 'formik';
import { Button } from '@yandex-data-ui/common';
import React, { PropsWithChildren, useCallback, useMemo, useRef, useState } from 'react';
import classes from '../../../../design/commonModals.module.css';
import { ScenarioParams } from '../../../../models';
import { DevJson } from '../../../../shared';
import { AdvancedField, KeyValueField, ProjectSelectField } from '../../../form_fields2';
import { TrackerIssueField } from '../../../form_fields2/TrackerIssueField';
import { reservedHostsTransferScenarioSchema, ScenarioType } from '../../models';
import { ScenarioNameField } from '../fields/ScenarioNameField';
import { ScenarioTypeField } from '../fields/ScenarioTypeField';
import { useSelector } from 'react-redux';
import { RootState } from 'state/store';

export const scenarioReservedHostsTransferFields: ExtendedFieldConfig<ScenarioFormFields>[] = [
   {
      as: ScenarioTypeField,
      label: 'Scenario type',
      name: 'scenario_type',
      required: true,
   },
   {
      as: TrackerIssueField,
      label: 'Ticket',
      name: 'ticket_key',
      required: true,
   },
   {
      as: ScenarioNameField,
      label: 'Name',
      name: 'name',
      placeholder: 'Enter scenario name',
      required: true,
   },
   {
      as: ProjectSelectField,
      label: 'Project',
      name: 'script_args.target_project_id' as any,
      required: true,
   },
   {
      as: SetField,
      controlProps: {
         type: 'string',
      },
      label: 'Hosts',
      name: 'hosts',
      placeholder: 'Enter host inventory numbers, FQDNs or UUIDs',
      required: true,
   },
   {
      as: KeyValueField,
      label: 'Labels',
      name: 'labels',
      placeholder: 'Enter labels',
   },
   {
      as: CheckboxField2,
      label: 'Autostart',
      name: 'autostart',
   },

   {
      as: TextField2,
      hint: 'Will be logged',
      label: 'Comment',
      name: 'reason',
      placeholder: 'Enter a reason for project creation',
   },

   // Advanced (only for admin)
   {
      as: AdvancedField,
      hint: '(only for admin)',
      label: 'Advanced options',
      name: '_advanced',
      hidden: (_, formikProps) => {
         return !formikProps.values._isAdmin;
      },
   },
   {
      as: InputField2,
      label: 'Idle time',
      hint: 'time in ms',
      name: 'script_args.idle_time' as any,
      placeholder: 'Enter time',
      controlProps: {
         type: 'number',
      },
      hidden: (_, formikProps) => {
         return !formikProps.values._advanced;
      },
   },
   {
      as: CheckboxField2,
      label: 'Workdays only',
      name: 'script_args.workdays_only' as any,
      hidden: (_, formikProps) => {
         return !formikProps.values._advanced;
      },
   },
];

type ScenarioFormFields = ScenarioParams & {
   _advanced: boolean;
   _isAdmin: boolean;
};

const initialValueBase: ScenarioFormFields = {
   name: '',
   hosts: new Set(),
   script_args: {
      delete: false,
   },
   labels: {},
   autostart: false,
   scenario_type: ScenarioType.ReservedHostsTransfer,
   _advanced: false,
   _isAdmin: false,
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <DeclarativeFields configs={scenarioReservedHostsTransferFields} />

      {children}
   </Form>
);

interface Props {
   initialValue?: Partial<ScenarioFormFields>;

   onSubmit(values: ScenarioParams, formikHelpers: FormikHelpers<ScenarioFormFields>): void;

   onScenarioTypeChange(value: string): void;

   onDismiss(): void;
}

export const CreateScenarioReservedHostsTransferForm: React.FC<Props> = React.memo(
   ({ onSubmit, onDismiss, onScenarioTypeChange, initialValue }) => {
      // hooks
      const user = useSelector((s: RootState) => s.globals.user);
      const isAdmin = user ? user.isAdmin : false;

      const [apiError, setApiError] = useState('');
      const formRef = useRef<FormikProps<ScenarioFormFields>>();

      const initialValuePatched = useMemo(() => ({ ...initialValueBase, _isAdmin: isAdmin, ...initialValue }), [
         initialValue,
         isAdmin,
      ]);

      const formHooks = useMemo(
         () =>
            ({
               onFieldChange: (name, value) => {
                  setApiError('');

                  if (!formRef.current) {
                     return;
                  }

                  if (name === 'scenario_type') {
                     if (typeof value !== 'string') {
                        throw new Error('Scenario type must be string');
                     }

                     onScenarioTypeChange(value);
                  }
               },
            } as FormHooks<ScenarioFormFields>),
         [onScenarioTypeChange, formRef],
      );

      const onFormSubmit = useCallback((values, formikHelpers) => onSubmit(values, formikHelpers), [onSubmit]);

      // render
      return (
         <Formik
            initialValues={initialValuePatched}
            onSubmit={onFormSubmit}
            validationSchema={reservedHostsTransferScenarioSchema}
         >
            {form => {
               formRef.current = form;

               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <InternalForm>
                        <DevJson>{{ form }}</DevJson>

                        <footer className={classes.footer}>
                           {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                           <Button
                              view={'clear'}
                              size={'m'}
                              className={classes.button}
                              onClick={onDismiss}
                              disabled={form.isSubmitting}
                           >
                              Cancel
                           </Button>

                           <Button
                              view={'action'}
                              size={'m'}
                              className={classes.button}
                              type={'submit'}
                              disabled={!form.isValid || form.isSubmitting || !form.dirty}
                           >
                              {form.values.autostart ? 'Create and start' : 'Create'}
                           </Button>
                        </footer>
                     </InternalForm>
                  </FormHooksContext.Provider>
               );
            }}
         </Formik>
      );
   },
);

CreateScenarioReservedHostsTransferForm.displayName = 'CreateScenarioReservedHostsTransferForm';
