import { ActionButtons, IAction } from '@yandex-infracloud-ui/libs-next';
import React, { SyntheticEvent, useCallback } from 'react';
import { IScenario } from '../../../../models';
import { ScenarioType } from '../../models';
import { CancelScenarioModal } from '../actions/CancelScenarioModal/CancelScenarioModal';
import { FinishScenarioModal } from '../actions/FinishScenarioModal/FinishScenarioModal';
import { PauseScenarioModal } from '../actions/PauseScenarioModal/PauseScenarioModal';
import { SkipScenarioModal } from '../actions/SkipScenarioModal/SkipScenarioModal';
import { StartScenarioModal } from '../actions/StartScenarioModal/StartScenarioModal';
import { useOpenModal } from '../config';
import { OperationStatusType } from '../OperationStatus/OperationStatus';

interface Props {
   item: IScenario;

   onCancel(): void;
}

interface ModalProps {
   item: IScenario;
}
const canCallCheckType = (action: string, item: IScenario) => {
   /**
    * finish: itdc, noc-hard, noc-soft
    * skip: hosts-transfer, reserved-host-transfer
    *
    * start: all
    * pause: all
    * cancel: all
    */

   switch (action) {
      case 'finish':
         return (
            item.scenario_type === ScenarioType.ITDCMaintenance ||
            item.scenario_type === ScenarioType.NOCHard ||
            item.scenario_type === ScenarioType.NOCSoft
         );

      case 'skip':
         return (
            item.scenario_type === ScenarioType.HostsTransfer ||
            item.scenario_type === ScenarioType.ReservedHostsTransfer
         );

      case 'start':
      case 'pause':
      case 'cancel':
         return true;

      default:
         return false;
   }
};

const canCallCheckStatus = (action: string, item: IScenario) => {
   /**
    * start: created, paused
    * pause: started
    * skip: paused
    * cancel: created, started, ready, paused
    * finish: ready
    */

   switch (action) {
      case 'start':
         return item.status === OperationStatusType.Created || item.status === OperationStatusType.Paused;

      case 'pause':
         return item.status === OperationStatusType.Started;

      case 'skip':
         return item.status === OperationStatusType.Paused;

      case 'cancel':
         return (
            item.status === OperationStatusType.Created ||
            item.status === OperationStatusType.Started ||
            item.status === OperationStatusType.Ready ||
            item.status === OperationStatusType.Paused
         );

      case 'finish':
         // TODO реализация бека, необходимо тут проверять WorkStatus
         return item.workStatus === OperationStatusType.Ready;

      default:
         return false;
   }
};

export const ScenarioActionButtons = ({ item, onCancel }: Props) => {
   // hooks
   const { openModal } = useOpenModal<ModalProps>();

   // handlers
   const handleStart = useCallback(() => openModal(StartScenarioModal, { item }), [item, openModal]);

   const handleFinish = useCallback(() => openModal(FinishScenarioModal, { item }), [item, openModal]);

   const handlePause = useCallback(() => openModal(PauseScenarioModal, { item }), [item, openModal]);

   const handleSkip = useCallback(() => openModal(SkipScenarioModal, { item }), [item, openModal]);

   const handleCancel = useCallback(() => openModal(CancelScenarioModal, { item }, onCancel), [
      item,
      onCancel,
      openModal,
   ]);

   const onAction = (e: SyntheticEvent, action: string) => {
      switch (action) {
         case 'start':
            handleStart();
            break;

         case 'pause':
            handlePause();
            break;

         case 'skip':
            handleSkip();
            break;

         case 'cancel':
            handleCancel();
            break;

         case 'finish':
            handleFinish();
            break;
      }
   };

   // Render
   // TODO: remove hardcoded actions
   const actionButtons: IAction[] = [
      { action: 'start', important: true, name: 'Start' },
      { action: 'pause', important: true, name: 'Pause' },
      { action: 'skip', important: true, name: 'Skip' },
      { action: 'cancel', important: true, name: 'Cancel' },
      { action: 'finish', important: true, name: 'Finish' },
   ];

   const actionsForType: IAction[] = actionButtons.filter(a => canCallCheckType(a.action, item));

   const disabledActions: Set<string> = new Set(
      actionsForType.filter(a => !canCallCheckStatus(a.action, item)).map(a => a.action),
   );

   return (
      <ActionButtons compact={false} actions={actionsForType} onAction={onAction} disabledActions={disabledActions} />
   );
};

ScenarioActionButtons.displayName = 'ScenarioActionButtons';
