import React, { useCallback } from 'react';
import { RadioButton, YCSelect } from '@yandex-data-ui/common';
import { arrayToYCSelectItem } from 'utils/toLibs';

import { IScenarioFilters } from '../../../../models';
import { TextInputSearch, useConstants } from '../../../../shared';
import { IssuerFilterType } from '../../models';
import classes from './ScenarioFilters.module.css';

interface Props {
   filters: Required<IScenarioFilters>;

   onChange(f: IScenarioFilters): void;
}

export const ScenarioFilters = React.memo(({ filters, onChange }: Props) => {
   // hooks
   const { constants } = useConstants();

   const ScenarioTypesFilters = arrayToYCSelectItem(constants ? constants.scenario_types : []);
   const ScenarioWorkStatusFilters = arrayToYCSelectItem(constants ? constants.scenario_work_statuses : []);

   // handlers
   const updateQuery = useCallback((q: string | number) => onChange({ ...filters, query: String(q) }), [
      filters,
      onChange,
   ]);

   const updateIssuer = useCallback((v: string) => onChange({ ...filters, issuer: v as IssuerFilterType }), [
      filters,
      onChange,
   ]);

   const updateWorkStatus = useCallback(
      (v: string[]) => {
         onChange({ ...filters, workStatus: new Set(v) });
      },
      [filters, onChange],
   );

   const updateType = useCallback(
      (v: string[]) => {
         onChange({ ...filters, type: new Set(v) });
      },
      [filters, onChange],
   );

   return (
      <div className={classes.filters} data-e2e={'ScenarioFilters'}>
         <div>
            <RadioButton onUpdate={updateIssuer} value={filters.issuer} name={'issuer'}>
               <RadioButton.Option value={IssuerFilterType.My} data-e2e={'my'}>
                  My
               </RadioButton.Option>
               <RadioButton.Option value={IssuerFilterType.All} data-e2e={'all'}>
                  All
               </RadioButton.Option>
            </RadioButton>
         </div>

         <YCSelect
            onUpdate={updateWorkStatus}
            className={classes.statusSelect}
            type={'multiple'}
            value={Array.from(filters.workStatus)}
            items={ScenarioWorkStatusFilters}
            label={'Status:'}
            placeholder={'All Statuses'}
         />

         <YCSelect
            onUpdate={updateType}
            className={classes.typeSelect}
            type={'multiple'}
            value={Array.from(filters.type)}
            items={ScenarioTypesFilters}
            label={'Type:'}
            showSearch={false}
            placeholder={'All Types'}
         />

         <TextInputSearch
            view={'normal'}
            type={'text'}
            placeholder={'Name or id'}
            className={classes.query}
            value={filters.query}
            hasClear={true}
            onUpdate={updateQuery}
         />
      </div>
   );
});
