import { DISMISS_REASON, FormFieldType, FormLayout, IFormField, useDismounted } from '@yandex-infracloud-ui/libs-next';
import { IModalProps, ModalLayout, toasts } from '@yandex-infracloud-ui/libs';
import { WarnIcon } from 'design/icons';
import * as React from 'react';
import { SyntheticEvent, useCallback, useState } from 'react';
import { takeUntil } from 'rxjs/operators';
import { IScenario, IScenarioCancelParams } from '../../../../../models';
import { scenariosApi } from '../../../../../services';
import classes from './CancelScenarioModal.module.css';

interface Props extends IModalProps<IScenarioCancelParams> {
   item: IScenario;
}

const fields: IFormField<IScenarioCancelParams>[] = [
   {
      label: 'Reason',
      name: 'reason',
      placeholder: 'Reason (optional)',
      type: FormFieldType.Text,
   },
];

export const CancelScenarioModal = React.memo(({ item, ok, cancel }: Props) => {
   // hooks
   const dismounted = useDismounted();
   const [value, setValue] = useState<IScenarioCancelParams>({
      reason: '',
   });

   // handlers
   const onOk = useCallback(() => {
      scenariosApi
         .cancel(item.scenario_id, value)
         .pipe(takeUntil(dismounted))
         .subscribe(() => ok(value), toasts.handleApiError('Cancel scenario'));
   }, [dismounted, item.scenario_id, ok, value]);

   const onCancel = useCallback(() => cancel(DISMISS_REASON), [cancel]);
   const updateParams = useCallback((e: SyntheticEvent | null, v: IScenarioCancelParams) => setValue(v), []);

   // render
   const subtitle = (
      <>
         <p className={classes.warning}>
            <WarnIcon size={'lg'} className={classes.icon} />
            Cancelling scenario will NOT cancel host tasks and will NOT perform any rollback operations.
            <br />
            You need to cancel host tasks and rollback any changes manually.
         </p>

         <p>Please, provide a reason for this operation.</p>
      </>
   );

   return (
      <ModalLayout title={'Cancel scenario'} subtitle={subtitle} onOk={onOk} onDismiss={onCancel}>
         <FormLayout onChange={updateParams} value={value} fields={fields} />
      </ModalLayout>
   );
});

CancelScenarioModal.displayName = 'CancelScenarioModal';
