import {
   FormFieldType,
   FormLayout,
   getValidResult,
   IFormField,
   IValidationResult,
   Loader,
   SaveFooter,
   SuggestLayer,
   useDismounted,
   validateEntity,
} from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useState } from 'react';
import { RouteComponentProps, StaticContext } from 'react-router';
import { takeUntil } from 'rxjs/operators';

import {
   IAutomationPlot,
   ICheck,
   plotSchema,
   plotSchemaWithOwners,
   plotShape,
   plotShapeWithOwners,
   ROUTE_LINKS,
} from '../../../models';
import { newOwnersField } from '../../../rich_shared/form_fields';
import { automationPlotsApi, config } from '../../../services';
import { HeaderWithBackLink, Page, useCanEditPlot } from '../../../shared';

import { CheckList } from '../components/CheckList';

import styles from './item.module.css';

const fields: IFormField<IAutomationPlot>[] = [
   {
      label: 'Plot name',
      name: 'name',
      placeholder: 'Enter plot name',
      required: true,
      type: FormFieldType.String,
   },
];

const fieldsInternal: IFormField<IAutomationPlot>[] = [
   ...fields,
   {
      component: newOwnersField(SuggestLayer.PeopleOrGroups),
      label: 'Owners',
      name: 'owners',
      type: FormFieldType.Custom,
   },
];
const fieldsExternal: IFormField<IAutomationPlot>[] = [
   ...fields,
   {
      label: 'Folder ID',
      name: 'yc_iam_folder_id',
      type: FormFieldType.String,
      required: true,
      placeholder: 'Enter a folder ID for YC iam',
   },
];

type Props = RouteComponentProps<{ id: string }, StaticContext, { prev: any }>;

export function PlotItem(props: Props) {
   const goToList = useCallback(() => props.history.push(ROUTE_LINKS.automationPlots()), [props]);

   const plotId = props.match.params.id;

   // hooks
   const dismounted = useDismounted();

   const [validation, setValidation] = useState<IValidationResult<IAutomationPlot>>(getValidResult());

   const [plot, setPlot] = useState<IAutomationPlot | null>(null);
   const canEdit = useCanEditPlot(plot);

   useEffect(() => {
      if (plot) {
         return;
      }

      automationPlotsApi
         .getById(plotId)
         .pipe(takeUntil(dismounted))
         .subscribe(
            resp => setPlot(resp),
            resp => {
               toasts.apiError('Automation plot loading', resp);
               goToList();
            },
         );
   }, [props, goToList, dismounted, plot, plotId]);

   const onCommonParametersChange = (e: SyntheticEvent, params: IAutomationPlot) => {
      setValidation(getValidResult());
      setPlot(params);
   };

   const onChecksChange = (checks: ICheck[]) =>
      setPlot({
         ...plot!,
         checks,
      });

   const save = useCallback(() => {
      const vr = validateEntity<Partial<IAutomationPlot>>(
         config.isExternal ? plotSchema : plotSchemaWithOwners,
         Object.keys(config.isExternal ? plotShape : plotShapeWithOwners) as (keyof IAutomationPlot)[],
         plot!,
         null,
      );

      if (!vr.isValid) {
         toasts.error('Please check form', 'Can not save');
         setValidation(vr);

         return;
      }

      automationPlotsApi
         .update(plot!)
         .pipe(takeUntil(dismounted))
         .subscribe(() => {
            toasts.success(`Automation plot ${plot!.name} has been updated`);
            setPlot(null);
         }, toasts.handleApiError('Automation plot creating'));
   }, [dismounted, plot]);

   return (
      <Page title={`Edit automation plot${plot ? ` "${plot.name}` : ''}`} className={styles.page}>
         <HeaderWithBackLink
            text={'Edit automation plot'}
            url={ROUTE_LINKS.automationPlots()}
            location={props.location}
         />

         {!plot ? (
            <Loader text={'Automation plot loading'} />
         ) : (
            <>
               <div className={styles.subHeader}>Common parameters</div>
               <div className={styles.formWrapper}>
                  <FormLayout
                     fields={config.isExternal ? fieldsExternal : fieldsInternal}
                     readonly={!canEdit}
                     value={plot!}
                     onChange={onCommonParametersChange}
                     validationResult={validation}
                  />
               </div>

               <div className={styles.subHeader}>
                  Checks and behaviour
                  <div className={styles.hint}>Checks are evaluated in the order bellow</div>
               </div>

               <CheckList readonly={!canEdit} plotId={plotId} value={plot!.checks!} onChange={onChecksChange} />
            </>
         )}

         {canEdit ? <SaveFooter onCancel={goToList} onSave={save} disabled={!plot || !validation.isValid} /> : null}
      </Page>
   );
}
