import { fromQuery, Loader, toQuery, useDismounted } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-data-ui/common';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useReducer } from 'react';
import { RouteComponentProps } from 'react-router';
import { takeUntil } from 'rxjs/operators';

import { ROUTE_LINKS } from '../../../models';
import { automationPlotsApi, config } from '../../../services';
import { Page, TextInputSearch } from '../../../shared';

import styles from './list.module.css';
import { PlotListItem } from './list_item';
import { ActionType, initialState, reducer } from './state';

export interface IUrlParams {
   q?: string;
}

export const PlotList = React.memo((props: RouteComponentProps<IUrlParams>) => {
   const urlParams: IUrlParams = fromQuery(props.location.search);

   // hooks
   const dismounted = useDismounted();
   const [{ filteredItems, isLoading }, dispatch] = useReducer(reducer, {
      ...initialState,
      q: urlParams.q || '',
   });

   useEffect(() => {
      dispatch({ type: ActionType.SetFilterQuery, q: urlParams.q });
   }, [urlParams.q]);

   useEffect(() => {
      automationPlotsApi
         .getList()
         .pipe(takeUntil(dismounted)) // Отписка, если компонент был уничтожен (повлечет отмену XHR-запроса)
         .subscribe(
            resp => dispatch({ type: ActionType.Loaded, items: resp.result }),
            resp => {
               dispatch({ type: ActionType.SetLoading, isLoading: false });
               toasts.apiError('Checks loading', resp);
            },
         );
   }, [dismounted]);

   // handlers
   const goToNew = useCallback(
      (e: SyntheticEvent) => {
         // Хак кнопки-ссылки LEGO, чтобы она не перегружала страницу
         e.preventDefault();

         props.history.push(ROUTE_LINKS.automationPlot('new'));
      },
      [props.history],
   );

   const updateQuery = useCallback(
      (v: string | number) => {
         const search = toQuery({ q: v });

         props.history.push({ search });
      },
      [props.history],
   );

   // render
   return (
      <Page title={'Check list'}>
         <div className={styles.topBar}>
            <TextInputSearch
               view={'normal'}
               type={'text'}
               placeholder={'Name or id'}
               className={styles.query}
               value={urlParams.q}
               hasClear={true}
               onUpdate={updateQuery}
            />

            <Button view={'action'} size={'m'} href={ROUTE_LINKS.automationPlot('new')} onClick={goToNew}>
               Create
            </Button>
         </div>

         <table className={styles.table}>
            <thead>
               <tr>
                  <th>Name</th>
                  {config.isExternal ? null : <th>Owners</th>}
                  <th />
               </tr>
            </thead>
            <tbody>
               {filteredItems.map(item => (
                  <PlotListItem
                     key={item.id}
                     location={props.location}
                     item={item}
                     onRemoved={id => dispatch({ type: ActionType.Removed, id })}
                  />
               ))}
            </tbody>
         </table>

         {filteredItems.length === 0 && !isLoading ? <div className={styles.noData}>No data</div> : null}

         <Loader text={'Check list loading'} visible={isLoading} cls={styles.loader} />
      </Page>
   );
});

PlotList.displayName = 'PlotList';
