import { hasIncluded } from '@yandex-infracloud-ui/libs-next';
import { IAutomationPlot } from '../../../models';

/**
 * Все возможные типы actions
 */
export enum ActionType {
   Loaded,
   Removed,
   SetFilterQuery,
   SetLoading,
}

interface IActionLoaded {
   items: IAutomationPlot[];
   type: ActionType.Loaded;
}

interface IActionRemoved {
   id: string;
   type: ActionType.Removed;
}

interface IActionSetFilterQuery {
   q: string | undefined;
   type: ActionType.SetFilterQuery;
}

interface IActionSetLoading {
   isLoading: boolean;
   type: ActionType.SetLoading;
}

type Action = IActionLoaded | IActionRemoved | IActionSetFilterQuery | IActionSetLoading;

export const initialState = {
   allItems: [] as IAutomationPlot[],
   filteredItems: [] as IAutomationPlot[],
   isLoading: true,
   q: '',
};

export type State = Readonly<typeof initialState>;

const filterItems = (items: IAutomationPlot[], q = '') => items.filter(item => hasIncluded(q, item.id, item.name));

// Классический redux-редьюсер. Должен возвращать стейт целиком.
export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.Loaded: {
         const allItems = action.items;

         return {
            ...state,
            allItems,
            filteredItems: filterItems(allItems, state.q),
            isLoading: false,
         };
      }

      case ActionType.Removed: {
         const allItems = state.allItems.filter(item => item.id !== action.id);

         return {
            ...state,
            allItems,
            filteredItems: filterItems(allItems, state.q),
         };
      }

      case ActionType.SetFilterQuery: {
         return {
            ...state,
            filteredItems: filterItems(state.allItems, action.q),
            q: action.q || '',
         };
      }

      case ActionType.SetLoading: {
         return {
            ...state,
            isLoading: action.isLoading,
         };
      }

      default: {
         return state;
      }
   }
};
