import { EMPTY_VALUE, IPureValueProps, isEmpty, styleHelpers } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { PropsWithChildren, useState } from 'react';
import { arrayMove, SortableContainer, SortOver } from 'react-sortable-hoc';

import { ICheck } from '../../../models';

import { CheckLayout } from './CheckLayout';
import styles from './CheckList.module.css';
import { CheckListItem } from './CheckListItem';
import { NewCheckListItem } from './NewCheckListItem';

const getEmptyCheck = (): ICheck => ({
   enabled: true,
   name: '',
   reboot: false,
   redeploy: false,
   profile: false,
   wait: false,
   report_failure: true,
});

interface IProps extends IPureValueProps<ICheck[]> {
   plotId?: string;
   readonly: boolean;
}

const SortableRows = SortableContainer(({ children }: PropsWithChildren<any>) => (
   <ul className={styleHelpers.unstyledList}>{children}</ul>
));

export const CheckList = ({ readonly, plotId, value, onChange }: IProps) => {
   const [newCheck, setNewCheck] = useState<ICheck>(getEmptyCheck());

   const onItemChange = (item: ICheck) => onChange(value.map(v => (v.name === item.name ? item : v)));

   const onItemDelete = (item: ICheck) => onChange(value.filter(v => v !== item));

   const addNewCheck = () => {
      if (!isEmpty(newCheck.name)) {
         onChange([...value, newCheck]);
         setNewCheck(getEmptyCheck());
      }
   };

   const onSortEnd = ({ oldIndex, newIndex }: SortOver) => onChange(arrayMove(value, oldIndex, newIndex));

   return (
      <>
         <ul className={styleHelpers.unstyledList}>
            <CheckLayout
               cls={styles.header}
               dragHandle={<div />}
               name={'Name'}
               actions={'Actions for CRIT'}
               buttons={null}
            />

            <CheckLayout dragHandle={<div />} name={'Embedded checks'} actions={<>{EMPTY_VALUE}</>} buttons={null} />
         </ul>

         <SortableRows onSortEnd={onSortEnd} useDragHandle={true} lockAxis={'y'} lockToContainerEdges={true}>
            <>
               {value.map((item, i) => (
                  <CheckListItem
                     key={item.name}
                     index={i}
                     readonly={readonly}
                     plotId={plotId}
                     value={item}
                     onChange={onItemChange}
                     onDelete={onItemDelete}
                  />
               ))}
            </>
         </SortableRows>

         {readonly ? null : (
            <>
               <ul className={styleHelpers.unstyledList}>
                  <NewCheckListItem value={newCheck} onChange={setNewCheck} onAdd={addNewCheck} />
               </ul>
            </>
         )}
      </>
   );
};
