import { classNames, IPureValueProps } from '@yandex-infracloud-ui/libs-next';
import React, { useCallback } from 'react';
import { SortableElement, SortableHandle } from 'react-sortable-hoc';

import { ICheck } from '../../../models';
import { RowButtons, useModals } from '../../../shared';
import { CheckActions } from './CheckActions';
import { CheckLayout } from './CheckLayout';

import styles from './CheckListItem.module.css';
import { modalConfigs, TurnModal } from './modals/config';

import { DeleteIcon, PowerOffIcon, BarsIcon, WarnIcon } from 'design/icons';

const DragHandle = SortableHandle(() => (
   <div className={styles.drag}>
      <BarsIcon />
   </div>
));

interface IProps extends IPureValueProps<ICheck> {
   /**
    * Необходим для сортировки мышкой
    */
   index?: number;
   plotId?: string;
   readonly: boolean;

   onDelete(value: ICheck): void;
}

export const CheckListItem = SortableElement(({ readonly, plotId, value, onChange, onDelete }: IProps) => {
   // hooks
   const { openModal } = useModals(modalConfigs, () => onTurnChange());

   // handlers
   const onTurnChange = useCallback(() => onChange({ ...value, enabled: !value.enabled }), [onChange, value]);

   const buttons = [
      {
         // start_time задается на бекенде => проверка уже создана
         handler: () =>
            value.start_time
               ? openModal(plotId!, TurnModal.Turn, !value.enabled, { name: value.name })
               : onTurnChange(),
         name: value.enabled ? 'Turn off' : 'Turn on',
         icon: <PowerOffIcon />,
      },
      {
         handler: () => onDelete(value),
         icon: <DeleteIcon />,
         name: 'Remove',
      },
   ];

   const name = (
      <span className={classNames(styles.name, { [styles.disabled]: !value.enabled })}>
         {value.enabled ? null : <WarnIcon />}
         {value.name}
      </span>
   );

   return (
      <CheckLayout
         dragHandle={readonly ? <div /> : <DragHandle />}
         name={name}
         actions={<CheckActions readonly={readonly} value={value} onChange={onChange} />}
         buttons={readonly ? null : <RowButtons buttons={buttons} />}
      />
   );
});
