import { IModalProps, ModalLayout } from '@yandex-infracloud-ui/libs';
import {
   classNames,
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   Loader,
   parseApiError,
   useDismounted,
   IApiError,
} from '@yandex-infracloud-ui/libs-next';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { Button } from '@yandex-data-ui/common';
import { Form, Formik, FormikHelpers } from 'formik';
import { takeUntil } from 'rxjs/operators';

import classes from '../../../../design/commonModals.module.css';
import { CheckForm, checkFormValidationSchema } from './models';
import { ReasonField } from '../../../../rich_shared/form_fields2/ReasonField';
import { automationPlotsApi } from '../../../../services';

const initialValue: CheckForm = {
   reason: '',
};

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <ReasonField />

      {children}
   </Form>
);

interface Props extends IModalProps<CheckForm> {
   id: string;
   name: string;
}

export const TurnOnModal: React.FC<Props> = ({ id, name, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<CheckForm>),
      [],
   );

   const handleSubmit = useCallback(
      (v: CheckForm, helpers: FormikHelpers<CheckForm>) => {
         automationPlotsApi
            .enableCheck(id, name, v.reason)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  ok(v);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, id, name, ok],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} showFooter={false} title={'You are going to turn on the check'}>
         <Formik initialValues={initialValue} onSubmit={handleSubmit} validationSchema={checkFormValidationSchema}>
            {form => (
               <FormHooksContext.Provider value={formHooks}>
                  <InternalForm>
                     <footer className={classes.footer}>
                        {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                        <Loader visible={form.isSubmitting} inline={true} text={'In progress'} />

                        <Button
                           view={'clear'}
                           size={'m'}
                           className={classes.button}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                        >
                           Cancel
                        </Button>

                        <Button
                           view={'action'}
                           size={'m'}
                           className={classNames(classes.button)}
                           type={'submit'}
                           disabled={!form.isValid || form.isSubmitting}
                        >
                           Turn on
                        </Button>
                     </footer>
                  </InternalForm>
               </FormHooksContext.Provider>
            )}
         </Formik>
      </ModalLayout>
   );
};

TurnOnModal.displayName = 'TurnOnModal';
