import {
   FormFieldType,
   FormLayout,
   getValidResult,
   IFormField,
   IValidationResult,
   SaveFooter,
   SuggestLayer,
   useDismounted,
   validateEntity,
} from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { SyntheticEvent, useCallback, useState } from 'react';
import { RouteComponentProps, StaticContext } from 'react-router';
import { takeUntil } from 'rxjs/operators';

import {
   globalState,
   IAutomationPlot,
   ICheck,
   plotSchema,
   plotShape,
   plotShapeWithOwners,
   plotSchemaWithOwners,
   ROUTE_LINKS,
} from '../../../models';
import { newOwnersField } from '../../../rich_shared/form_fields';
import { automationPlotsApi, config } from '../../../services';
import { HeaderWithBackLink, Page } from '../../../shared';
import { idFromName } from '../../../utils';

import { CheckList } from '../components/CheckList';
import styles from './new.module.css';

const fields: IFormField<IAutomationPlot>[] = [
   {
      label: 'Plot name',
      name: 'name',
      placeholder: 'Enter plot name',
      required: true,
      type: FormFieldType.String,
   },
   {
      label: 'Plot ID',
      name: 'id',
      placeholder: 'Enter plot ID',
      required: true,
      type: FormFieldType.String,
   },
];
const fieldsInternal: IFormField<IAutomationPlot>[] = [
   ...fields,
   {
      component: newOwnersField(SuggestLayer.PeopleOrGroups),
      label: 'Owners',
      name: 'owners',
      type: FormFieldType.Custom,
   },
];
const fieldsExternal: IFormField<IAutomationPlot>[] = [
   ...fields,
   {
      label: 'Folder ID',
      name: 'yc_iam_folder_id',
      type: FormFieldType.String,
      required: true,
      placeholder: 'Enter a folder ID for YC iam',
   },
];

export function NewPlot(props: RouteComponentProps<{}, StaticContext, { prev: any }>) {
   const goToPlotByID = useCallback(id => props.history.push(ROUTE_LINKS.automationPlot(id)), [props]);
   const goToList = useCallback(() => props.history.push(ROUTE_LINKS.automationPlots()), [props]);

   const dismounted = useDismounted();

   const [validation, setValidation] = useState<IValidationResult<IAutomationPlot>>(getValidResult());

   const [autoIdEnabled, setAutoIdEnabled] = useState(true);

   const [plot, setPlot] = useState<IAutomationPlot>({
      checks: [],
      id: '',
      name: '',
      owners: config.isExternal ? undefined : new Set(globalState.user.value ? [globalState.user.value.login] : []),
      yc_iam_folder_id: undefined,
   });

   const setAutoId = (params: IAutomationPlot) => {
      const hasIdChanged = plot.id !== params.id;
      if (!hasIdChanged && autoIdEnabled) {
         params.id = idFromName(params.name);
      } else {
         // Если имя было изменено вручную, больше не будет его генерировать
         setAutoIdEnabled(false);
      }
   };

   const onCommonParametersChange = (e: SyntheticEvent, params: IAutomationPlot) => {
      if (autoIdEnabled) {
         setAutoId(params);
      }
      setValidation(getValidResult());
      setPlot(params);
   };

   const onChecksChange = (checks: ICheck[]) =>
      setPlot({
         ...plot,
         checks,
      });

   const save = useCallback(() => {
      const vr = validateEntity<Partial<IAutomationPlot>>(
         config.isExternal ? plotSchema : plotSchemaWithOwners,
         Object.keys(config.isExternal ? plotShape : plotShapeWithOwners) as (keyof IAutomationPlot)[],
         plot,
         null,
      );

      if (!vr.isValid) {
         toasts.error('Please check form', 'Can not save');
         setValidation(vr);

         return;
      }

      automationPlotsApi
         .create(plot)
         .pipe(takeUntil(dismounted))
         .subscribe(res => {
            toasts.success(`Automation plot ${plot.name} has been created`);
            goToPlotByID(res.id);
         }, toasts.handleApiError('Automation plot creating'));
   }, [plot, dismounted, goToPlotByID]);

   return (
      <Page title={'New automation plot'} className={styles.page}>
         <HeaderWithBackLink
            text={'New automation plot'}
            url={ROUTE_LINKS.automationPlots()}
            location={props.location}
         />

         <div className={styles.subHeader}>Common parameters</div>
         <div className={styles.formWrapper}>
            <FormLayout
               fields={config.isExternal ? fieldsExternal : fieldsInternal}
               value={plot}
               onChange={onCommonParametersChange}
               validationResult={validation}
            />
         </div>

         <div className={styles.subHeader}>
            Checks and behaviour
            <div className={styles.hint}>Checks are evaluated in the order bellow</div>
         </div>

         <CheckList readonly={false} value={plot.checks!} onChange={onChecksChange} />

         <SaveFooter saveText={'Create plot'} onCancel={goToList} onSave={save} disabled={!validation.isValid} />
      </Page>
   );
}
