import { capitalize, IDismountedProps, Loader, withDismounted, ExternalLink } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import { number, oneOf } from 'prop-types';
import * as React from 'react';
import { takeUntil } from 'rxjs/operators';

import { EXTERNAL_LINKS, HostLogType } from '../../../models';
import { hostApi } from '../../../services';
import styles from './HostLog.module.css';

/**
 * Обрезает мусор в начале лога
 *
 * Так как API позволяет ограничить хвост лога только байтами,
 * то иногда в начале лога прилетают обрывки предыдущих строк.
 * Вот их-то мы и откасываем
 */
export function _trimPartLog(text: string): string {
   // Дата в начале строки в форматах: 2018-06-25 или 2018.06.25
   const dateRe = /^\d\d\d\d[.-]\d\d[.-]\d\d/m;

   const i = text.search(dateRe);

   return i < 0 ? text : text.substring(i);
}

interface IProps {
   hostUuid: string;
   type: HostLogType;
   hostName?: string;
}

interface IState {
   isLoading: boolean;
   log: string;
}

class HostLog extends React.PureComponent<IProps & IDismountedProps, IState> {
   public static propTypes = {
      hostInv: number.isRequired,
      type: oneOf([HostLogType.Deploy, HostLogType.Profile, HostLogType.Coroner]).isRequired,
   };

   public state: IState = {
      isLoading: false,
      log: '',
   };

   public componentDidMount(): void {
      this.setState({ isLoading: true }, () => {
         switch (this.props.type) {
            case HostLogType.Coroner:
               hostApi
                  .getCoronerLogTail(this.props.hostName!)
                  .pipe(takeUntil(this.props.dismounted!))
                  .subscribe(
                     log => this.setState({ log: _trimPartLog(log), isLoading: false }),
                     err => {
                        toasts.apiError('Log loading', err);
                        this.setState({ isLoading: false });
                     },
                  );
               break;
            case HostLogType.Deploy:
            case HostLogType.Profile:
            default:
               hostApi
                  .getLogTail(this.props.hostUuid, this.props.type)
                  .pipe(takeUntil(this.props.dismounted!))
                  .subscribe(
                     log => this.setState({ log: _trimPartLog(log), isLoading: false }),
                     err => {
                        toasts.apiError('Log loading', err);
                        this.setState({ isLoading: false });
                     },
                  );
               break;
         }
      });
   }

   public render() {
      if (this.state.isLoading) {
         return <Loader text={`${capitalize(this.props.type)} log loading`} />;
      }

      const fullLogUrl =
         this.props.type === HostLogType.Coroner
            ? EXTERNAL_LINKS.coronerLog(this.props.hostName!)
            : hostApi.getLogLink(this.props.hostUuid, this.props.type);

      return (
         <>
            <div className={styles.fullLink}>
               <ExternalLink target={'_blank'} href={fullLogUrl}>
                  Show full {this.props.type} log
               </ExternalLink>
            </div>

            <pre className={styles.hostLogPre}>{this.state.log}</pre>
         </>
      );
   }
}

export const HostLogEnhanced = withDismounted(HostLog);
