import { Json, Loader, PageError, useObservableValue, WindowTitle } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useState } from 'react';
import { EXTERNAL_LINKS } from '../../../models';

import { healthApi } from '../../../services/api/health_api';
import { HostContext } from '../../../shared';

import { HostHealthTable } from './components/HostHealthTable';
import styles from './HostHealthTab.module.css';
import { RadioButton } from '@yandex-data-ui/common';

interface IProps {
   jugglerAggregateName?: string;
   name: string;
}

enum ViewMode {
   Table = 'table',
   Json = 'json',
}

const HostHealthTab = React.memo(({ jugglerAggregateName, name }: IProps) => {
   // hooks
   const [viewMode, setViewMode] = useState(ViewMode.Table);
   const viewOptions = React.useMemo(
      () => [
         {
            content: 'Table',
            value: ViewMode.Table,
         },
         {
            content: 'JSON',
            value: ViewMode.Json,
         },
      ],
      [],
   );

   const raw = useObservableValue(healthApi.getHostHealthDetails(name), toasts.handleApiError('Checks loading'), [
      name,
   ]);

   // handlers
   const toggle = useCallback(() => setViewMode(viewMode === ViewMode.Json ? ViewMode.Table : ViewMode.Json), [
      viewMode,
      setViewMode,
   ]);

   // render

   return (
      <>
         <div className={styles.header}>
            <RadioButton className={styles.tumbler} onChange={toggle} options={viewOptions} value={viewMode} />
            {jugglerAggregateName ? (
               <a
                  href={EXTERNAL_LINKS.jugglerAggregationChecks(jugglerAggregateName)}
                  target={'_blank'}
                  rel={'noopener noreferrer'}
               >
                  Checks in Juggler
               </a>
            ) : null}
         </div>

         {viewMode === ViewMode.Json ? (
            <div className={styles.json}>
               <Json obj={raw} />
            </div>
         ) : (
            <HostHealthTable value={raw} />
         )}
      </>
   );
});

HostHealthTab.displayName = 'HostHealthTab';

export function HostHealthTabConsumed() {
   return (
      <HostContext.Consumer>
         {value =>
            value.isLoading ? (
               <Loader text={'Host loading'} />
            ) : (
               <>
                  {value.host ? (
                     <>
                        <WindowTitle value={`Host ${value.host.name} health check`} />
                        <HostHealthTab
                           name={value.host.name}
                           jugglerAggregateName={value.host.juggler_aggregate_name}
                        />
                     </>
                  ) : (
                     <PageError text={value.error} />
                  )}
               </>
            )
         }
      </HostContext.Consumer>
   );
}
