import { classNames, formatDate, fromTimestamp, Json, toggleSetItem } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { useCallback, useState } from 'react';

import { HELP_HINTS, IHostHealthRecord, IListResult } from '../../../../models';
import { ClickOnlyUnmovedTrap, HealthCheckList } from '../../../../shared';

import styles from './HostHealthTable.module.css';

interface IProps {
   value: IListResult<IHostHealthRecord> | null;
}

export const HostHealthTable = React.memo(({ value }: IProps) => {
   if (!value) {
      return null;
   }
   const [expanded, setExpanded] = useState(new Set<string>()); // eslint-disable-line react-hooks/rules-of-hooks

   // hooks
   // eslint-disable-next-line react-hooks/rules-of-hooks
   const toggleExpanded = useCallback(
      (e: React.MouseEvent<HTMLTableRowElement>) => {
         setExpanded(toggleSetItem(expanded, e.currentTarget.dataset.id as string));
      },
      [expanded],
   );

   // render
   return (
      <table className={styles.table}>
         <thead>
            <tr>
               <th>Type</th>
               <th>Timestamp</th>
               <th>Status</th>
               <th>Status time</th>
            </tr>
         </thead>
         <tbody>
            {value.result.map(record => (
               <React.Fragment key={record.id}>
                  <ClickOnlyUnmovedTrap
                     as={'tr'}
                     onClick={toggleExpanded}
                     className={classNames(styles.row, {
                        [styles.expanded]: expanded.has(record.id),
                     })}
                     data-id={record.id}
                  >
                     <td>
                        <HealthCheckList inline={true} items={[{ name: record.type, status: record.status }]} />
                     </td>
                     <td>{formatDate(fromTimestamp(record.timestamp))}</td>
                     <td>
                        {record.status}: {HELP_HINTS.healthCheckStatuses[record.status]}
                     </td>
                     <td>{formatDate(fromTimestamp(record.status_mtime))}</td>
                  </ClickOnlyUnmovedTrap>

                  {expanded.has(record.id) ? (
                     <tr className={styles.additional} data-id={record.id}>
                        <td colSpan={4}>
                           <Json obj={record.metadata} />
                        </td>
                     </tr>
                  ) : null}
               </React.Fragment>
            ))}
         </tbody>
      </table>
   );
});

HostHealthTable.displayName = 'HostHealthTable';
