import { TabItemProps, Tabs } from '@yandex-data-ui/common';
import { autobind, CopyableText, IDismountedProps, isEqual, withDismounted } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { SyntheticEvent } from 'react';
import { Route, RouteComponentProps, StaticContext, Switch } from 'react-router';
import { of } from 'rxjs';
import { takeUntil } from 'rxjs/operators';

import { HostAction, HostActionButtons } from '../../actions/host_actions';
import {
   globalState,
   IApiError,
   IHost,
   parseApiError,
   ProjectType,
   ROUTE_LINKS,
   SLOW_UPDATE_INTERVAL,
} from '../../models';
import { auth, dictApi, hostApi } from '../../services';
import {
   HeaderWithBackLink,
   HostContext,
   HostLinks,
   IHostContextValue,
   Page,
   RequestHostAccessLink,
} from '../../shared';

import { HostStatusDetails } from './components/HostStatusDetails';
import styles from './index.module.css';
import { HOST_TABS, SHADOW_HOST_TABS } from './routes';

interface IPageParams {
   id: string;
}

type Props = RouteComponentProps<IPageParams, StaticContext, { prev: any }>;

class HostScreen extends React.Component<Props & IDismountedProps, IHostContextValue> {
   public static defaultProps = {};

   public static propTypes = {};

   private readonly _tabs: TabItemProps[];
   private readonly _tabsShadow: TabItemProps[];

   private _updateTimerId: number | null = null;

   constructor(props: Props) {
      super(props);

      const hostName = this.props.match.params.id;

      this._tabs = HOST_TABS.map(
         r =>
            ({
               id: (r.path as string).replace(':id', hostName),
               title: r.name ? r.name.toUpperCase() : r.name,
            } as TabItemProps),
      );

      this._tabsShadow = SHADOW_HOST_TABS.map(
         r =>
            ({
               id: (r.path as string).replace(':id', hostName),
               title: r.name ? r.name.toUpperCase() : r.name,
            } as TabItemProps),
      );

      this.state = {
         boxes: null,
         canEdit: false,
         canRunHostAction: false,
         error: '',
         host: null,
         isLoading: false,
         isShadow: false,
         onChange: (e: SyntheticEvent | null, host: IHost) => {
            this.setState({ host });
         },
      };
   }

   public componentDidMount(): void {
      this._loadHost();
      this._loadBoxes();

      globalState.user.pipe(takeUntil(this.props.dismounted!)).subscribe(user =>
         this.setState({
            canEdit: auth.canEditHost(this.state.host, user),
            canRunHostAction: auth.canRunHostAction(this.state.host, user),
         }),
      );
   }

   public componentWillUnmount(): void {
      this._unscheduleUpdating();
   }

   public render() {
      const { id } = this.props.match.params;
      const { location, history } = this.props;
      const routes = (this.state.isShadow ? SHADOW_HOST_TABS : HOST_TABS).map(r => (
         <Route key={r.path as string} {...r} />
      ));

      const header = (
         <div className={styles.header}>
            <div className={styles.title}>
               <HeaderWithBackLink
                  text={this.state.host && this.state.host.name ? this.state.host.name : id}
                  url={ROUTE_LINKS.home()}
                  location={location}
               />
               &nbsp;
               <span className={styles.copyIcon}>
                  <CopyableText text={this.state.host ? this.state.host.name : id} />
               </span>
            </div>

            <div className={styles.actions}>
               {this.state.host ? <RequestHostAccessLink host={this.state.host} /> : null}

               <HostActionButtons
                  compact={true}
                  currentContext={this.state.host ? [this.state.host] : []}
                  getSelectedHosts={() => of([this.state.host!])}
                  disabled={!this.state.canRunHostAction}
                  onActionFinished={this._onActionFinished}
                  isShadow={this.state.isShadow}
               />
            </div>

            <HostStatusDetails host={this.state.host} location={location} />

            <div className={styles.links}>
               {this.state.host ? (
                  <HostLinks
                     name={this.state.host.name}
                     inv={this.state.host.inv}
                     project={this.state.host.project}
                     boxes={this.state.boxes}
                  />
               ) : null}
            </div>

            <div className={styles.tabs}>
               <Tabs
                  activeTab={location.pathname}
                  items={this.state.isShadow ? this._tabsShadow : this._tabs}
                  onSelectTab={path => history.push(path)}
               />
            </div>
         </div>
      );

      return (
         <Page header={header} title={`Host ${this.props.match.params.id}`}>
            <HostContext.Provider value={this.state}>
               <Switch>{routes}</Switch>
            </HostContext.Provider>
         </Page>
      );
   }

   private _loadHost() {
      this._unscheduleUpdating();

      this.setState({ isLoading: true }, () => {
         hostApi
            .getByFQDN(this.props.match.params.id)
            .pipe(takeUntil(this.props.dismounted!))
            .subscribe(
               host =>
                  this.setState(
                     {
                        canEdit: auth.canEditHost(host),
                        canRunHostAction: auth.canRunHostAction(host),
                        host,
                        isLoading: false,
                        isShadow: host.type === ProjectType.SHADOW,
                     },
                     () => this._scheduleUpdating(),
                  ),
               (resp: IApiError) => {
                  const error = `Host loading error: ${parseApiError(resp)}`;

                  this.setState({ isLoading: false, error, canEdit: false, canRunHostAction: false });
                  toasts.apiError('Host loading', resp);
               },
            );
      });
   }

   private _loadBoxes() {
      dictApi
         .getBoxes()
         .pipe(takeUntil(this.props.dismounted!))
         .subscribe(boxes => this.setState({ boxes }));
   }

   @autobind
   private _onActionFinished(action: HostAction): void {
      if (action === HostAction.Remove) {
         this.props.history.push(ROUTE_LINKS.home());

         return;
      }

      this._loadHost();
   }

   private _scheduleUpdating(): void {
      this._unscheduleUpdating();

      this._updateTimerId = window.setTimeout(() => this._updateHost(), SLOW_UPDATE_INTERVAL);
   }

   private _unscheduleUpdating(): void {
      if (this._updateTimerId) {
         window.clearTimeout(this._updateTimerId);
         this._updateTimerId = null;
      }
   }

   private _updateHost(): void {
      hostApi
         .getByFQDN(this.props.match.params.id)
         .pipe(takeUntil(this.props.dismounted!))
         .subscribe(host => {
            if (!isEqual(host, this.state.host)) {
               this.setState({ host }, () => this._scheduleUpdating());
            } else {
               this._scheduleUpdating();
            }
         }, toasts.handleApiError('Host updating'));
   }
}

export const HostScreenEnhanced = withDismounted(HostScreen);
