import { Loader, PageError, WindowTitle } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { RouteComponentProps } from 'react-router';

import { IHost } from '../../../models';
import { HostContext } from '../../../shared';
import { HostBotDetails } from './components/HostBotDetails';
import { HostConfigurationDetails } from './components/HostConfigurationDetails';
import { HostDeployDetails } from './components/HostDeployDetails';
import { HostHealthDetails } from './components/HostHealthDetails';
import { HostLocationDetails } from './components/HostLocationDetails';
import { HostMaintenanceDetails } from './components/HostMaintenanceDetails';
import { HostPowerStatusDetails } from './components/HostPowerStatusDetails';
import { HostVlanDetails } from './components/HostVlanDetails';
import { Section } from './components/Section';

import styles from './HostInformation.module.css';

interface IProps extends RouteComponentProps {
   host: IHost;
   isShadow: boolean;
}

export const HostInformation = React.memo(({ host, isShadow }: IProps) => {
   return (
      <>
         <div className={styles.page}>
            <div className={styles.column}>
               <Section title={'Location'} collapsed={false}>
                  <HostLocationDetails host={host} />
               </Section>

               <Section title={'Configuration'} collapsed={false}>
                  <HostConfigurationDetails host={host} />
               </Section>

               {isShadow === false ? (
                  <Section title={'Deploy'} collapsed={false}>
                     <HostDeployDetails
                        config={host.config}
                        deploy_config_policy={host.deploy_config_policy}
                        network={host.deploy_network}
                        provisioner={host.provisioner}
                        tags={host.deploy_tags}
                     />
                  </Section>
               ) : null}
            </div>

            <div className={styles.column}>
               {host.state_expire ? (
                  <Section title={'Maintenance details'} collapsed={false}>
                     <HostMaintenanceDetails value={host.state_expire} />
                  </Section>
               ) : null}
               {isShadow === false && host.health ? (
                  <Section title={'Host health'} collapsed={false}>
                     <HostHealthDetails fqdn={host.name} health={host.health} />
                  </Section>
               ) : null}
               <Section title={'BOT information'} collapsed={true}>
                  <HostBotDetails inv={host.inv} />
               </Section>
               <Section title={'Current VLAN configuration'} collapsed={true}>
                  <HostVlanDetails host={host} />
               </Section>
               {isShadow === false ? (
                  <Section title={'IPMI Power status'} collapsed={true}>
                     <HostPowerStatusDetails host={host} />
                  </Section>
               ) : null}
               {/*TODO Extra VLANs
            <Section title='Extra parameters' collapsed={false}>
               <HostExtraDetails host={host}/>
            </Section>*/}
            </div>
         </div>
      </>
   );
});

HostInformation.displayName = 'HostInformation';

export function HostInformationConsumed(routeProps: RouteComponentProps) {
   return (
      <HostContext.Consumer>
         {value =>
            value.isLoading ? (
               <Loader text={'Host loading'} />
            ) : (
               <>
                  {value.host ? (
                     <>
                        <WindowTitle value={`Host ${value.host.name} information`} />
                        <HostInformation host={value.host} isShadow={value.isShadow} {...routeProps} />
                     </>
                  ) : (
                     <PageError text={value.error} />
                  )}
               </>
            )
         }
      </HostContext.Consumer>
   );
}
