import { CopyableText, EMPTY_VALUE, formatDate, styleHelpers } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';

import { EXTERNAL_LINKS, IHost, IHostLocation } from '../../../../models';
import { Field } from './Field';

import styles from './HostLocationDetails.module.css';

interface IProps {
   host: IHost;
}

function ShortVariant({ value, title }: { value: string; title: string }) {
   if (!value) {
      return null;
   }

   return (
      <>
         &nbsp;
         <span className={styles.short} title={title}>
            {value}
         </span>
      </>
   );
}

export const HostLocationDetails = React.memo(({ host }: IProps) => {
   const location = host.location || ({} as IHostLocation);
   const infinibandInfo = host.infiniband_info || null;

   const physicalLocationString = [location.country, location.city, location.datacenter, location.queue, location.rack]
      .filter(i => i)
      .join('|');

   const locationTitle = (
      <>
         Physical location
         {physicalLocationString ? (
            <>
               <br />
               <CopyableText text={physicalLocationString} />
            </>
         ) : null}
      </>
   );

   return (
      <>
         <Field title={''} customTitle={locationTitle}>
            {location.country ? <div>Country: {location.country}</div> : null}

            {location.city ? <div>City: {location.city}</div> : null}

            {location.datacenter ? (
               <div>
                  Data center: {location.datacenter}
                  <ShortVariant title={'Short datacenter name'} value={location.short_datacenter_name} />
               </div>
            ) : null}

            {location.queue ? (
               <div>
                  Queue: {location.queue}
                  <ShortVariant title={'Short queue name'} value={location.short_queue_name} />
               </div>
            ) : null}

            {location.rack ? <div>Rack: {location.rack}</div> : null}

            {physicalLocationString ? null : EMPTY_VALUE}
         </Field>

         <Field title={'Update time'}>
            {location.physical_timestamp ? formatDate(location.physical_timestamp as Date) : EMPTY_VALUE}
         </Field>

         <Field title={'Network location'}>
            {!location.switch && !location.port ? (
               EMPTY_VALUE
            ) : (
               <>
                  <div>Switch: {location.switch ? location.switch : EMPTY_VALUE}</div>

                  <div>
                     Port:{' '}
                     {location.switch && location.port ? (
                        <a
                           href={EXTERNAL_LINKS.rackTables(location.switch, location.port)}
                           className={styleHelpers.externalLink}
                           target={'_blank'}
                           rel={'noopener noreferrer'}
                        >
                           {location.port}
                        </a>
                     ) : location.port ? (
                        location.port
                     ) : (
                        EMPTY_VALUE
                     )}
                  </div>
               </>
            )}
         </Field>

         <Field title={'Actualization time'}>
            {location.network_timestamp ? formatDate(location.network_timestamp as Date) : EMPTY_VALUE}
         </Field>

         <Field title={'Information source'}>{location.network_source ? location.network_source : EMPTY_VALUE}</Field>

         <Field title={'Infiniband info'}>
            {!infinibandInfo ? (
               EMPTY_VALUE
            ) : (
               <>
                  <div>Cluster tag: {infinibandInfo.cluster_tag ?? EMPTY_VALUE}</div>
                  <div>Ports: {infinibandInfo.ports ? infinibandInfo.ports.join(' | ') : EMPTY_VALUE}</div>
               </>
            )}
         </Field>
      </>
   );
});

HostLocationDetails.displayName = 'HostLocationDetails';
