import { classNames, Loader } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import React, { useEffect, useState } from 'react';
import { catchError, tap } from 'rxjs/operators';
import { ReactComponent as PowerOffSvg } from '../../../../design/icons/power-off.svg';
import { ReactComponent as PowerOnSvg } from '../../../../design/icons/power-on.svg';
import { IHost, PendingStatus } from '../../../../models';
import { hostApi } from '../../../../services';

import classes from './HostPowerStatusDetails.module.css';

interface Props {
   host: IHost;
}

export const HostPowerStatusDetails = React.memo(({ host }: Props) => {
   const [pending, setPending] = useState(PendingStatus.New);

   const [isPoweredOn, setIsPoweredOn] = useState<boolean | null>(null);

   useEffect(() => {
      setPending(PendingStatus.Loading);
      const s = hostApi
         .getPowerStatus(host.uuid)
         .pipe(
            tap(() => setPending(PendingStatus.Successful)),
            catchError(err => {
               setPending(PendingStatus.Error);

               throw err;
            }),
         )
         .subscribe(setIsPoweredOn, toasts.handleApiError('Power status loading'));

      return () => s.unsubscribe();
   }, [host.uuid]);

   switch (pending) {
      default:
         return <Loader align={'left'} cls={classes.loader} />;

      case PendingStatus.Error:
         return <div className={classes.error}>Error loading status</div>;

      case PendingStatus.Successful:
         return (
            <div
               className={classNames(classes.result, {
                  [classes.on]: isPoweredOn,
                  [classes.off]: !isPoweredOn,
               })}
            >
               {isPoweredOn ? (
                  <>
                     <PowerOnSvg />
                     on
                  </>
               ) : (
                  <>
                     <PowerOffSvg />
                     off
                  </>
               )}
            </div>
         );
   }
});

HostPowerStatusDetails.displayName = 'HostPowerStatusDetails';
