import { Loader, useDismounted } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useEffect, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { IHost, IVlanConfig } from '../../../../models';
import { hostApi } from '../../../../services';
import { HbfProjectView } from '../../../../shared/HbfProjectInput/HbfProjectView';

import { Field } from './Field';
import styles from './HostVlanDetails.module.css';
import { VlanTable } from './VlanTable';

interface IProps {
   host: IHost;
}

const SyncStatus = ({ synced }: { synced?: boolean }) => {
   if (synced === undefined) {
      return null;
   }

   return synced ? (
      <span className={styles.synced}>(synced)</span>
   ) : (
      <span className={styles.notSynced}>(not synced)</span>
   );
};

export const HostVlanDetails = React.memo(({ host }: IProps) => {
   // hooks
   const dismounted = useDismounted();
   const [vlanConfig, setVlanConfig] = useState<IVlanConfig | null>(null);

   // effects
   useEffect(() => {
      hostApi
         .getVLANs(host.uuid)
         .pipe(takeUntil(dismounted))
         .subscribe(setVlanConfig, toasts.handleApiError('VLANs configuration loading'));
   }, [dismounted, host.uuid]);

   // render
   if (!vlanConfig) {
      return <Loader text={'Loading VLAN Configuration'} />;
   }

   return (
      <>
         <section className={styles.section}>
            <h4>
               VLANS <SyncStatus synced={vlanConfig.vlans_synced} />
            </h4>

            <Field title={'Current'} fullWidth={true}>
               <VlanTable selected={vlanConfig.native_vlan} list={vlanConfig.vlans} />
            </Field>

            {vlanConfig.expected_native_vlan ? (
               <Field title={'Expected'} fullWidth={true}>
                  <VlanTable selected={vlanConfig.expected_native_vlan} list={vlanConfig.expected_vlans} />
               </Field>
            ) : null}
         </section>

         <section className={styles.section}>
            <h4>
               MTN project ID <SyncStatus synced={vlanConfig.hbf_project_id_synced} />
            </h4>

            <Field title={'Current'} fullWidth={true}>
               <HbfProjectView id={vlanConfig.hbf_project_id} />
            </Field>

            <Field title={'Expected'} fullWidth={true}>
               <HbfProjectView id={vlanConfig.expected_hbf_project_id} />
            </Field>
         </section>
      </>
   );
});

HostVlanDetails.displayName = 'HostVlanDetails';
