import { CopyableText } from '@yandex-infracloud-ui/libs-next';
import { string } from 'prop-types';
import * as React from 'react';

import { MacAddress } from '../../../../models';

import styles from './MacAddressPopup.module.css';

interface IProps {
   value: MacAddress;
}

export class MacAddressPopup extends React.PureComponent<IProps> {
   public static defaultProps = {};

   public static propTypes = {
      value: string.isRequired,
   };

   private readonly _formats: (string | null)[];

   constructor(props: IProps) {
      super(props);

      this._formats = [
         formatMacAddress(this.props.value, 'canonical'),
         formatMacAddress(this.props.value, 'short'),
         formatMacAddress(this.props.value, 'integer'),
      ];
   }

   public render() {
      const isInvalid = this._formats[0] === null;

      if (isInvalid) {
         return (
            <div className={styles.macAddressPopup}>
               <div className={styles.invalid}>Invalid MAC address</div>
            </div>
         );
      }

      return (
         <div className={styles.macAddressPopup}>
            <h4>Copy MAC address</h4>

            {this._formats.map(mac => (
               <div key={mac!} className={styles.macAddress}>
                  {mac}
                  <CopyableText text={mac!} />
               </div>
            ))}
         </div>
      );
   }
}

type MacFormat = 'canonical' | 'short' | 'integer';

/**
 * Форматирует MAC-адрес, полученный из API в разные представления:
 * canonical   00:25:90:83:4e:53    (в lower case)
 * short       0025.9083.4e53       (отличается только точками)
 * integer     161338314323         (в виде 10-base integer)
 *
 * Код на python (с backend):
 * @example
 *    def mac_to_int(mac):
 *       return int(mac.replace(".", "").replace(":", ""), 16)
 *
 * @param address {MacAddress} строка с MAC адресом
 * @param format ожидаемый формат адреса
 */
export function formatMacAddress(address: MacAddress, format: MacFormat): string | null {
   const fullRe = /^([\da-f]{2}[:.]){5}[\da-f]{2}$/i;
   const shortRe = /^([\da-f]{4}\.){2}[\da-f]{4}$/i;

   if (!fullRe.test(address) && !shortRe.test(address)) {
      return null;
   }

   switch (format) {
      case 'canonical':
         return address.toString().toLowerCase();

      case 'short':
         const p = address.split(':');

         return `${p[0]}${p[1]}.${p[2]}${p[3]}.${p[4]}${p[5]}`;

      case 'integer':
         return parseInt(address.replace(/[:.]/g, ''), 16).toString();

      default:
         console.warn(`Unexpected MAC format: ${format}`);

         return address;
   }
}
