import {
   arrayToSet,
   isEqual,
   IValueProps,
   Loader,
   PageError,
   SaveFooter,
   useDismounted,
   WindowTitle,
} from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { SyntheticEvent, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { hostActionService } from '../../../actions/host_actions';
import { IHost, IHostActionParams } from '../../../models';
import { hostApi } from '../../../services';
import { HostContext, RestrictionEditor } from '../../../shared';

import styles from './HostRestrictions.module.css';

interface IProps extends IValueProps<IHost> {
   canEdit: boolean;
}

const HostRestrictions = React.memo(({ canEdit, value, onChange }: IProps) => {
   const initialRestrictions = arrayToSet(value.restrictions);

   // hooks
   const dismounted = useDismounted();
   const [restrictions, setRestrictions] = useState(initialRestrictions);
   const [changed, setChanged] = useState(false);

   // helpers
   const _save = (params: IHostActionParams) => {
      hostApi
         .updateRestrictions(value.uuid, { ...params, restrictions })
         .pipe(takeUntil(dismounted))
         .subscribe(() => {
            toasts.success('Restrictions has been updated');
            setChanged(false);
            onChange(null, { ...value, restrictions });
         }, toasts.handleApiError('Host restrictions saving'));
   };

   // handlers
   const askSave = () => {
      hostActionService
         .askBaseParams('Restrictions updating')
         .pipe(takeUntil(dismounted))
         .subscribe(
            _save,
            () => undefined, // просто закрыли модалку
         );
   };

   const updateRestrictions = (e: SyntheticEvent | null, r: Set<string>) => {
      setRestrictions(r);
      setChanged(!isEqual(r, initialRestrictions));
   };

   const cancel = () => setRestrictions(initialRestrictions);

   // render
   return (
      <div className={styles.hostRestrictions}>
         <div className={styles.form}>
            <RestrictionEditor disabled={!canEdit} value={restrictions} onChange={updateRestrictions} />
         </div>

         {canEdit ? <SaveFooter onSave={askSave} onCancel={cancel} disabled={!changed} /> : null}
      </div>
   );
});

HostRestrictions.displayName = 'HostRestrictions';

export function HostRestrictionsConsumed() {
   return (
      <HostContext.Consumer>
         {value =>
            value.isLoading ? (
               <Loader text={'Host loading'} />
            ) : (
               <>
                  {value.host ? (
                     <>
                        <WindowTitle value={`Host ${value.host.name} restrictions`} />
                        <HostRestrictions value={value.host} onChange={value.onChange} canEdit={value.canEdit} />
                     </>
                  ) : (
                     <PageError text={value.error} />
                  )}
               </>
            )
         }
      </HostContext.Consumer>
   );
}
