import {
   useDismounted,
   DeclarativeFields,
   ExtendedFieldConfig,
   FormHooks,
   FormHooksContext,
   FormContext,
   Loader,
   setAsArray,
} from '@yandex-infracloud-ui/libs-next';
import { modalService, toasts } from '@yandex-infracloud-ui/libs';
import { RouteComponentProps, StaticContext } from 'react-router';
import { Form, Formik, FormikProps } from 'formik';
import React, { PropsWithChildren, useCallback, useMemo, useRef, useState, useEffect } from 'react';
import classes from '../../../design/commonModals.module.css';
import { object, string, array, number } from 'yup';
import { HeaderWithBackLink, Page, DevJson, useCanEditPlot, ReasonModal } from '../../../shared';
import { IMaintenancePlot, ROUTE_LINKS } from '../../../models';

import styles from './MaintenancePlotItem.module.css';
import { MetaInfo } from '../components/MetaInfo';
import { CommonSettings } from '../components/CommonSettings';
import { ScenarioSettings } from '../components/ScenarioSettings';
import { idFromName } from '../../../utils';
import { maintenancePlotsApi } from 'services/api/maintenance_plots_api';
import { takeUntil } from 'rxjs/operators';
import { Card, Button } from '@yandex-data-ui/common';

export const scenarioSettingsShape = {
   scenario_type: string().required().nullable(),
};

const maintenancePlotShape = {
   id: string().required('This is a required field').nullable(),
   meta_info: object({
      abc_service_slug: string().required('This is a required field').nullable(),
      name: string().required('This is a required field').nullable(),
   }),
   common_settings: object({
      common_scenario_settings: object({
         dont_allow_start_scenario_if_total_number_of_active_hosts_more_than: number()
            .min(40, 'Min value 40')
            .nullable(),
         total_number_of_active_hosts: number().min(0, 'Min value 0').nullable(),
      }),
      maintenance_approvers: object({
         logins: setAsArray().of(string()),
         abc_roles_codes: setAsArray().of(string()),
         abc_role_scope_slugs: setAsArray().of(string()),
      }),
   }),
   scenarios_settings: array().of(object().shape(scenarioSettingsShape)),
};

export const maintenancePlotSchema = object().shape(maintenancePlotShape);

const fields: ExtendedFieldConfig<IMaintenancePlot>[] = [
   {
      as: MetaInfo,
      label: 'MetaInfo',
      name: 'meta_info' as any,
      readonly: (_, formikProps) => {
         return !formikProps.values._canEdit && !formikProps.values._isNewPlot;
      },
   },
   {
      as: CommonSettings,
      label: 'Common Settings',
      name: 'common_settings' as any,
      readonly: (_, formikProps) => {
         return !formikProps.values._canEdit && !formikProps.values._isNewPlot;
      },
   },
   {
      as: ScenarioSettings,
      label: 'Scenario Settings',
      name: 'scenarios_settings',
      readonly: (_, formikProps) => {
         return formikProps.values.gc_enabled || (!formikProps.values._canEdit && !formikProps.values._isNewPlot);
      },
   },
];

function onKeyDown(keyEvent: any) {
   if ((keyEvent.charCode || keyEvent.keyCode) === 13) {
      keyEvent.preventDefault();
   }
}

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={styles.content} onKeyDown={onKeyDown}>
      <DeclarativeFields configs={fields} />

      {children}
   </Form>
);

export const MaintenancePlotItem = (props: RouteComponentProps<{ id: string }, StaticContext, { prev: any }>) => {
   const goToPlotByID = useCallback(id => props.history.push(ROUTE_LINKS.maintenancePlot(id)), [props]);
   const goToList = useCallback(() => props.history.push(ROUTE_LINKS.maintenancePlots()), [props]);

   const plotId = props.match.params.id;
   const isNewPlot = !plotId;

   const dismounted = useDismounted();

   const [actualPlot, setActualPlot] = useState<IMaintenancePlot | null>(null);

   const canEdit = useCanEditPlot(actualPlot);

   const [apiError, setApiError] = useState('');
   const formRef = useRef<FormikProps<IMaintenancePlot>>();

   const [autoIdEnabled, setAutoIdEnabled] = useState(isNewPlot);

   const initialValueBase: IMaintenancePlot = useMemo(
      () => ({
         _canEdit: canEdit,
         _isNewPlot: isNewPlot,
         id: '',
         meta_info: {
            abc_service_slug: '',
            name: '',
         },
         common_settings: {
            maintenance_approvers: {
               logins: [],
               abc_roles_codes: [],
               abc_role_scope_slugs: [],
               abc_duty_schedule_slugs: [],
            },
         },
         scenarios_settings: [],
      }),
      [canEdit, isNewPlot],
   );

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: (name, value) => {
               setApiError('');

               if (!formRef.current) {
                  return;
               }

               if (name === ('id' as any)) {
                  // Если ID был изменен вручную, больше не будет его генерировать
                  setAutoIdEnabled(false);
               }
               if (name === ('meta_info.name' as any) && autoIdEnabled) {
                  formRef.current.setFieldValue('id', idFromName(value as string));
               }

               if (name.includes('settings.use_yp_sla') && value) {
                  const scenarioSettingsItem = name.replace('.use_yp_sla', ''); // scenarios_settings[index].settings

                  formRef.current.setFieldValue(
                     `${scenarioSettingsItem}.request_cms_x_seconds_before_maintenance_start_time`,
                     null,
                  );
               }

               if (name.includes('settings.start_power_off_x_seconds_before_maintenance_start_time') && value) {
                  const scenarioSettingsItem = name.replace(
                     '.start_power_off_x_seconds_before_maintenance_start_time',
                     '',
                  ); // scenarios_settings[index].settings

                  formRef.current.setFieldValue(
                     `${scenarioSettingsItem}.enable_manual_approval_after_hosts_power_off`,
                     false,
                  );
               }

               if (name.includes('settings.enable_manual_approval_after_hosts_power_off') && value) {
                  const scenarioSettingsItem = name.replace('.enable_manual_approval_after_hosts_power_off', ''); // scenarios_settings[index].settings

                  formRef.current.setFieldValue(
                     `${scenarioSettingsItem}.start_power_off_x_seconds_before_maintenance_start_time`,
                     undefined,
                  );
               }
            },
         } as FormHooks<IMaintenancePlot>),
      [formRef, autoIdEnabled],
   );

   const handleSubmit = useCallback(
      (plot: IMaintenancePlot) => {
         if (isNewPlot) {
            delete plot._canEdit;
            delete plot._isNewPlot;

            plot.scenarios_settings = plot.scenarios_settings.map(e => {
               delete e.id;

               return e;
            });

            return new Promise((resolve, reject) => {
               maintenancePlotsApi
                  .create(plot)
                  .pipe(takeUntil(dismounted))
                  .subscribe(res => {
                     toasts.success(`Maintenance plot ${plot.meta_info.name} has been created`);
                     goToPlotByID(res.id);
                     // Formik refresh form
                     resolve(res.id);
                  }, toasts.handleApiError('Maintenance plot creating'));
            });
         } else {
            return new Promise((resolve, reject) => {
               modalService
                  .open(ReasonModal, { reason: 'Reason' })
                  .pipe(takeUntil(dismounted))
                  .subscribe(
                     ({ reason = '' }) => {
                        maintenancePlotsApi
                           .update(actualPlot!, plot!, reason)
                           .pipe(takeUntil(dismounted))
                           .subscribe(() => {
                              toasts.success(`Maintenance plot ${plot.meta_info.name} has been updated`);
                              setActualPlot(null);
                              // Formik refresh form
                              resolve(plot.id);
                           }, toasts.handleApiError('Maintenance plot creating'));
                     },
                     () => null,
                  );
            });
         }
      },
      [goToPlotByID, dismounted, isNewPlot, actualPlot],
   );

   useEffect(() => {
      if (isNewPlot || actualPlot) {
         return;
      }

      maintenancePlotsApi
         .getById(plotId)
         .pipe(takeUntil(dismounted))
         .subscribe(
            resp => setActualPlot(resp),
            resp => {
               toasts.apiError('Maintenance plot loading', resp);
               goToList();
            },
         );
   }, [props, goToList, dismounted, actualPlot, plotId, isNewPlot]);

   // render
   return (
      <Page title={isNewPlot ? 'New maintenance plot' : 'Edit maintenance plot'} className={styles.page}>
         <div style={{ display: 'flex', marginBottom: '1.75rem' }}>
            <Card theme={'warning'}>
               <div style={{ padding: 10, maxWidth: 650 }}>
                  <strong>Early access</strong>
                  <br />
                  Be aware that alpha versions may be less stable than final version officially released to the public.
                  You may encounter problems such as app crashes ro features that do not work properly.
               </div>
            </Card>
         </div>

         {!actualPlot && !isNewPlot ? (
            <Loader text={'Maintenance plot loading'} />
         ) : (
            <>
               <HeaderWithBackLink
                  text={isNewPlot ? 'New maintenance plot' : 'Edit maintenance plot'}
                  url={ROUTE_LINKS.maintenancePlots()}
                  location={props.location}
               />

               <Formik
                  initialValues={
                     isNewPlot ? initialValueBase : { ...actualPlot!, _canEdit: canEdit, _isNewPlot: isNewPlot }
                  }
                  onSubmit={handleSubmit}
                  validationSchema={maintenancePlotSchema}
                  enableReinitialize={true}
               >
                  {form => {
                     formRef.current = form;

                     return (
                        <FormHooksContext.Provider value={formHooks}>
                           <FormContext.Provider value={form}>
                              <InternalForm>
                                 <DevJson>{{ form }}</DevJson>

                                 <br />

                                 <footer className={classes.footer}>
                                    {apiError && !form.isSubmitting ? (
                                       <span className={classes.error}>{apiError}</span>
                                    ) : null}

                                    <Button
                                       view={'clear'}
                                       size={'m'}
                                       className={classes.button}
                                       onClick={goToList}
                                       disabled={form.isSubmitting}
                                    >
                                       Cancel
                                    </Button>

                                    <Button
                                       view={'action'}
                                       size={'m'}
                                       className={classes.button}
                                       type={'submit'}
                                       disabled={
                                          !form.isValid || form.isSubmitting || !form.dirty || (!canEdit && !isNewPlot)
                                       }
                                    >
                                       {isNewPlot ? 'Create Plot' : 'Save changes'}
                                    </Button>
                                 </footer>
                              </InternalForm>
                           </FormContext.Provider>
                        </FormHooksContext.Provider>
                     );
                  }}
               </Formik>
            </>
         )}
      </Page>
   );
};

MaintenancePlotItem.displayName = 'MaintenancePlotItem';
