import { Card, Button } from '@yandex-data-ui/common';
import { fromQuery, Loader, toQuery, useDismounted } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';

import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useReducer } from 'react';
import { RouteComponentProps } from 'react-router';
import { takeUntil } from 'rxjs/operators';

import { ROUTE_LINKS, EXTERNAL_LINKS } from '../../../models';
import { maintenancePlotsApi } from '../../../services';
import { Page, TextInputSearch } from '../../../shared';

import styles from './MaintenancePlotList.module.css';
import { MaintenancePlotListItem } from './MaintenancePlotListItem';
import { ActionType, initialState, reducer } from './state';

export interface IUrlParams {
   q?: string;
}

export const MaintenancePlotList = React.memo((props: RouteComponentProps<IUrlParams>) => {
   const urlParams: IUrlParams = fromQuery(props.location.search);

   // hooks
   const dismounted = useDismounted();
   const [{ filteredItems, isLoading }, dispatch] = useReducer(reducer, {
      ...initialState,
      q: urlParams.q || '',
   });

   useEffect(() => {
      dispatch({ type: ActionType.SetFilterQuery, q: urlParams.q });
   }, [urlParams.q]);

   useEffect(() => {
      maintenancePlotsApi
         .getList()
         .pipe(takeUntil(dismounted)) // Отписка, если компонент был уничтожен (повлечет отмену XHR-запроса)
         .subscribe(
            resp => dispatch({ type: ActionType.Loaded, items: resp.result }),
            resp => {
               dispatch({ type: ActionType.SetLoading, isLoading: false });
               toasts.apiError('Checks loading', resp);
            },
         );
   }, [dismounted]);

   // handlers
   const goToNew = useCallback(
      (e: SyntheticEvent) => {
         // Хак кнопки-ссылки LEGO, чтобы она не перегружала страницу
         e.preventDefault();

         props.history.push(ROUTE_LINKS.maintenancePlot('new'));
      },
      [props.history],
   );

   const goToDraft = useCallback(() => {
      window.open(EXTERNAL_LINKS.walleDraft('maintenance-plot-proektov'), '_blank');
   }, []);

   const updateQuery = useCallback(
      (v: string | number) => {
         const search = toQuery({ q: v });

         props.history.push({ search });
      },
      [props.history],
   );

   // render
   return (
      <Page title={'Maintenance plots'}>
         <div style={{ display: 'flex', marginBottom: '1.75rem' }}>
            <Card theme={'info'}>
               <div style={{ padding: 10, maxWidth: 650 }}>
                  <strong>New feature</strong>
                  <br />
                  We have added a new Maintenance Plot entity to Wall-E. This is a container of settings that will be
                  applied to the user's hosts in scenarios.
                  <br />
                  <br />
                  <Button type={'button'} view={'action'} onClick={goToDraft}>
                     Read Draft
                  </Button>
               </div>
            </Card>
         </div>

         <div className={styles.topBar}>
            <TextInputSearch
               view={'normal'}
               type={'text'}
               placeholder={'Name or id'}
               className={styles.query}
               value={urlParams.q}
               hasClear={true}
               onUpdate={updateQuery}
            />

            <Button view={'action'} size={'m'} href={ROUTE_LINKS.maintenancePlot('new')} onClick={goToNew}>
               Create
            </Button>
         </div>

         <table className={styles.table}>
            <thead>
               <tr>
                  <th>Name</th>
                  <th>ABC Service</th>
                  <th />
               </tr>
            </thead>
            <tbody>
               {filteredItems.map(item => (
                  <MaintenancePlotListItem
                     key={item.id}
                     location={props.location}
                     item={item}
                     onRemoved={id => dispatch({ type: ActionType.Removed, id })}
                  />
               ))}
            </tbody>
         </table>

         {filteredItems.length === 0 && !isLoading ? <div className={styles.noData}>No data</div> : null}

         <Loader text={'Check list loading'} visible={isLoading} cls={styles.loader} />
      </Page>
   );
});

MaintenancePlotList.displayName = 'MaintenancePlotList';
