import React, { useState } from 'react';
import { Tabs, TabsDirection, RadioButton } from '@yandex-data-ui/common';
import { classNames, ExtendedFieldConfig, useExtendedField } from '@yandex-infracloud-ui/libs-next';
import { IMaintenanceScenarioSetting } from '../../../models/api_models/maintenance_plots';
import { ScenarioType } from 'rich_shared/scenario_list/models';
import { ScenarioSettingsItem } from './ScenarioSettingsItem';

import classes from './ScenarioSettings.module.css';

const isIncludeScenarioType = (v: IMaintenanceScenarioSetting[], type: ScenarioType): boolean =>
   v.some(e => e.scenario_type === type) || false;

export const ScenarioSettings = (props: ExtendedFieldConfig<any, IMaintenanceScenarioSetting[], {}>) => {
   const { disabled, field, readonly, onChange } = useExtendedField(props);
   const [activeTab, setActiveTab] = useState(ScenarioType.ITDCMaintenance);

   const isIncludeITDC = isIncludeScenarioType(field.value, ScenarioType.ITDCMaintenance);
   const isIncludeNocHard = isIncludeScenarioType(field.value, ScenarioType.NOCHard);

   const onAdd = (scenarioType: ScenarioType) => () => {
      onChange([
         ...field.value,
         {
            scenario_type: scenarioType,
            settings: {},
         },
      ]);
   };
   const onAddITDC = onAdd(ScenarioType.ITDCMaintenance);
   const onAddNocHard = onAdd(ScenarioType.NOCHard);

   const onRemove = (scenarioType: ScenarioType) => () => {
      onChange([...field.value.filter(e => e.scenario_type !== scenarioType)]);
   };
   const onRemoveITDC = onRemove(ScenarioType.ITDCMaintenance);
   const onRemoveNocHard = onRemove(ScenarioType.NOCHard);

   return (
      <div className={classes.root}>
         <h2>Scenario Settings</h2>
         <div className={classes.wrapper}>
            <Tabs
               className={classes.tabs}
               direction={TabsDirection.Vertical}
               items={[
                  {
                     id: ScenarioType.ITDCMaintenance,
                     title: (
                        <span
                           className={classNames({ [classes.disableText]: !isIncludeITDC })}
                           data-e2e={'ITDCMaintenanceTab'}
                        >
                           {ScenarioType.ITDCMaintenance}
                        </span>
                     ),
                  },
                  {
                     id: ScenarioType.NOCHard,
                     title: (
                        <span
                           className={classNames({ [classes.disableText]: !isIncludeNocHard })}
                           data-e2e={'NOCHardTab'}
                        >
                           {ScenarioType.NOCHard}
                        </span>
                     ),
                  },
               ]}
               activeTab={activeTab}
               onSelectTab={(tabId: ScenarioType) => setActiveTab(tabId)}
            />

            <div className={classes.itemWrapper}>
               {activeTab === ScenarioType.ITDCMaintenance && (
                  <RadioButton
                     className={classes.radio}
                     options={[
                        {
                           content: 'Enable',
                           value: 'enable',
                        },
                        {
                           content: 'Disable',
                           value: 'disable',
                        },
                     ]}
                     value={isIncludeITDC ? 'enable' : 'disable'}
                     onUpdate={v => {
                        if (v === 'enable') {
                           onAddITDC();
                        } else {
                           onRemoveITDC();
                        }
                     }}
                     disabled={readonly || disabled}
                  />
               )}

               {activeTab === ScenarioType.NOCHard && (
                  <RadioButton
                     options={[
                        {
                           content: 'Enable',
                           value: 'enable',
                        },
                        {
                           content: 'Disable',
                           value: 'disable',
                        },
                     ]}
                     value={isIncludeNocHard ? 'enable' : 'disable'}
                     onUpdate={v => {
                        if (v === 'enable') {
                           onAddNocHard();
                        } else {
                           onRemoveNocHard();
                        }
                     }}
                     disabled={readonly || disabled}
                  />
               )}

               {field.value.map((el, index) => (
                  <ScenarioSettingsItem
                     key={el.scenario_type}
                     isHidden={el.scenario_type !== activeTab}
                     readonly={readonly}
                     disabled={disabled}
                     value={el}
                     index={index}
                     name={field.name}
                  />
               ))}
            </div>
         </div>
      </div>
   );
};
