import { AppRoute, Loader, styleHelpers, useDismounted } from '@yandex-infracloud-ui/libs-next';
import { modalService, toasts } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-data-ui/common';
import * as React from 'react';
import { useCallback, useEffect, useMemo, useReducer, useRef } from 'react';
import { Route, RouteComponentProps, StaticContext, Switch } from 'react-router';
import { Subscription } from 'rxjs';
import { finalize, takeUntil } from 'rxjs/operators';
import { PreorderContext } from 'shared/contexts';

import { EXTERNAL_LINKS, IPreorder } from '../../../models';
import { preorderApi } from '../../../services';
import { HeaderWithBackLink, ITab, Page, Tabs } from '../../../shared';

import { PreorderProgressBar } from '../components/PreorderProgressBar';
import { PreorderResetModal } from '../components/PreorderResetModal';
import { PREORDER_TABS } from '../routes';
import { PreorderDetails } from './components/PreorderDetails';
import styles from './item.module.css';
import { ActionType, initialState, reducer } from './item.state';

interface IMatchParams {
   id: string;
}

export const PreorderItem = React.memo(
   ({ match, location }: RouteComponentProps<IMatchParams, StaticContext, { prev: any }>) => {
      const preorderId = match.params.id;

      const routes = PREORDER_TABS.map(r => <Route key={r.path as string} {...r} />);

      // hooks
      const [{ isLoading, preorder }, dispatch] = useReducer(reducer, initialState);
      const dismounted = useDismounted();
      const previousRequest = useRef<Subscription>();

      // helpers
      const loadPreorder = useCallback(() => {
         dispatch({ type: ActionType.BeforeLoading });

         if (previousRequest.current) {
            previousRequest.current.unsubscribe();
         }

         previousRequest.current = preorderApi
            .getById(parseInt(preorderId, 10))
            .pipe(
               takeUntil(dismounted),
               finalize(() => (previousRequest.current = undefined)),
            )
            .subscribe(
               item => {
                  dispatch({ type: ActionType.Loaded, preorder: item });
               },
               err => {
                  dispatch({ type: ActionType.SetLoading, isLoading: false });
                  toasts.apiError('Preorder loading', err);
               },
            );
      }, [dismounted, preorderId]);

      // effects
      useEffect(() => {
         loadPreorder();
      }, [dismounted, loadPreorder, preorderId]);

      // handlers
      const onReset = useCallback(() => {
         modalService.open(PreorderResetModal, {}).subscribe(
            params => {
               preorderApi.restart(parseInt(preorderId, 10), params).subscribe(() => {
                  toasts.success(`Preorder #${preorderId} has been restarted`);
                  loadPreorder();
               }, toasts.handleApiError('Restart preorder'));
            },
            () => null, // just close
         );
      }, [loadPreorder, preorderId]);

      // render
      const getTabCount = (route: AppRoute, p: IPreorder) => {
         // TODO как-то некрасиво в коде обращаться к тексту в UI
         if (route.name === 'Errors') {
            return p.errors ? p.errors.length : 0;
         }

         if (route.name === 'Messages') {
            return p.messages ? p.messages.length : 0;
         }

         return undefined;
      };

      const tabs = PREORDER_TABS.map(
         r =>
            ({
               count: getTabCount(r, preorder),
               header: r.name ? r.name.toUpperCase() : r.name,
               id: r.path,
               url: (r.path as string).replace(':id', preorderId),
            } as ITab),
      );

      const header = (
         <div className={styles.header}>
            <div className={styles.titleWrapper}>
               <div className={styles.title}>
                  <HeaderWithBackLink url={''} text={`Preorder #${preorderId}`} location={location} />
               </div>

               <div className={styles.actions}>
                  <Button view={'outlined'} size={'m'} disabled={preorder.processed} onClick={onReset}>
                     Restart
                  </Button>
               </div>
            </div>

            <div className={styles.bar}>
               <PreorderProgressBar success={preorder.acquired_hosts || []} failed={preorder.failed_hosts || []} />
            </div>

            <PreorderDetails preorder={preorder} />

            <div className={styles.tabs}>
               <Tabs items={tabs} prev={location.state ? location.state.prev : null}>
                  <a
                     href={EXTERNAL_LINKS.botPreorder(preorderId)}
                     target={'_blank'}
                     rel={'noopener noreferrer'}
                     className={styleHelpers.externalGrayLink}
                  >
                     Bot
                  </a>
               </Tabs>
            </div>
         </div>
      );

      const contextValue = useMemo(
         () => ({
            acquired_hosts: preorder.acquired_hosts || [],
            errors: preorder.errors || [],
            failed_hosts: preorder.failed_hosts || [],
            isLoading,
            messages: preorder.messages || [],
            preorderId,
         }),
         [isLoading, preorder, preorderId],
      );

      return (
         <Page title={`Pre-order #${preorderId}`}>
            {header}
            {isLoading ? (
               <Loader text={'Pre-order is loading'} />
            ) : (
               <PreorderContext.Provider value={contextValue}>
                  <Switch>{routes}</Switch>
               </PreorderContext.Provider>
            )}
         </Page>
      );
   },
);

PreorderItem.displayName = 'PreorderItem';
