import { fromQuery, Loader, PER_PAGE_DEFAULT, queryValueToSet, setToQueryValue } from '@yandex-infracloud-ui/libs-next';
import { modalService, Pagination, toasts } from '@yandex-infracloud-ui/libs';
import { IPreorder } from 'models';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useRef } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { RootState } from 'state/store';

import { preorderApi } from '../../../services';
import { SidebarDirection, SidebarToggleButton, useTitleForUnselectedProjects } from '../../../shared';
import { idSetToSet } from '../../../state/commonModels';
import { projectsSlice } from '../../../state/projects';
import { patchUrlQuery } from '../../../utils/toLibs';

import { PreorderListWrapper } from '../components/PreorderListWrapper';
import { PreorderResetModal } from '../components/PreorderResetModal';
import { IPreorderUrlParams } from '../models';
import styles from './list.module.css';
import { PreorderListItem } from './list_item';
import { loadAll, setParamsFromProjects, setParamsFromUrl } from './preordersSlice';

export const PreorderList = React.memo(({ location, history, match }: RouteComponentProps) => {
   const urlParams = fromQuery(location.search) as IPreorderUrlParams;
   const page = urlParams.page ? parseInt(urlParams.page, 10) : 1;
   const perPage = urlParams.perPage ? parseInt(urlParams.perPage, 10) : PER_PAGE_DEFAULT;

   useTitleForUnselectedProjects('Pre-orders');

   // region hooks
   const {
      preorders: { filteredItems, isLoading },
      selectedProjects,
      userProjects,
   } = useSelector((state: RootState) => ({
      selectedProjects: state.projects.selectedIds,
      userProjects: state.globals.user && state.globals.user.source === 'api' ? state.globals.user.projects : undefined,
      preorders: state.preorders,
   }));
   const dispatch = useDispatch();
   const firstRun = useRef(true);
   const locationRef = useRef(location);
   // endregion

   // region effects
   useEffect(() => {
      // Первоначальная загрузка
      dispatch(loadAll());
   }, [dispatch]);

   useEffect(() => {
      // Обновление стейта после обновления URL
      dispatch(setParamsFromUrl(location.search));
   }, [location.search, dispatch]);

   useEffect(() => {
      locationRef.current = location;
   }, [location]);

   useEffect(() => {
      // Обновление списка при смене выбранных проектов
      const projectSet = idSetToSet(selectedProjects);

      dispatch(
         setParamsFromProjects({
            projects: projectSet,
            userProjects: userProjects || [],
         }),
      );

      patchUrlQuery(history, locationRef.current, {
         project: setToQueryValue(projectSet),
      } as IPreorderUrlParams);
   }, [userProjects, selectedProjects, dispatch, history]);

   useEffect(() => {
      if (!firstRun.current) {
         return;
      }

      dispatch(projectsSlice.actions.select(Array.from(queryValueToSet(urlParams.project))));
      firstRun.current = false;
   }, [dispatch, urlParams.project]);
   // endregion

   // region handlers
   const onReset = useCallback(
      (e: SyntheticEvent, id: number) => {
         modalService.open(PreorderResetModal, {}).subscribe(
            params => {
               preorderApi.restart(id, params).subscribe(() => {
                  toasts.success(`Preorder #${id} has been restarted`);
                  dispatch(loadAll());
               }, toasts.handleApiError('Restart preorder'));
            },
            () => null, // just close
         );
      },
      [dispatch],
   );

   const handlePageChange = useCallback(
      (e: SyntheticEvent | null, v: number) =>
         patchUrlQuery(history, locationRef.current, {
            page: v === 1 ? '' : v.toString(),
         }),
      [history],
   );

   const handlePerPageChange = useCallback(
      (e: SyntheticEvent | null, v: number) =>
         patchUrlQuery(history, locationRef.current, {
            perPage: v === PER_PAGE_DEFAULT ? '' : v.toString(),
         }),
      [history],
   );
   // endregion

   // region render
   return (
      <PreorderListWrapper location={location} history={history} match={match}>
         <table className={styles.table}>
            <thead>
               <tr>
                  <th className={styles.fixedColumn}>ID</th>
                  <th className={styles.fixedColumn}>Owner</th>
                  {/*<th className={styles.fixedColumn}>Status</th> TODO status*/}
                  <th className={styles.fixedColumn} />
                  <th />
               </tr>
            </thead>
            <tbody>
               {filteredItems.slice((page - 1) * perPage, page * perPage).map((preorder: IPreorder) => (
                  <PreorderListItem key={preorder.id} item={preorder} location={location} onReset={onReset} />
               ))}
            </tbody>
         </table>

         {filteredItems.length === 0 && !isLoading ? <div className={styles.noData}>No data</div> : null}

         <Loader text={'Preorders list loading'} visible={isLoading} cls={styles.loader} />

         <footer className={styles.footer}>
            <SidebarToggleButton className={styles.sidebarToggleButton} direction={SidebarDirection.Expand} />

            <div className={styles.paginationWrapper}>
               <div className={styles.pagination}>
                  <Pagination
                     perPage={perPage}
                     total={Math.ceil(filteredItems.length / perPage)}
                     value={page}
                     onChange={handlePageChange}
                     onPerPageChange={handlePerPageChange}
                  />
               </div>
            </div>
         </footer>
      </PreorderListWrapper>
   );
   // endregion
});

PreorderList.displayName = 'PreorderList';
