import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { fromQuery, queryValueToSet, sortHandler } from '@yandex-infracloud-ui/libs-next';
import { toasts } from '@yandex-infracloud-ui/libs';
import { map } from 'rxjs/operators';

import { DefaultOwnership, IListResult, IPreorder, OwnershipValue } from '../../../models';
import { preorderApi } from '../../../services';
import { AppThunk } from '../../../state/store';

import { IPreorderUrlParams } from '../models';

const initialState = {
   allItems: [] as IPreorder[],
   filteredItems: [] as IPreorder[],
   isLoading: true,
   projects: new Set<string>(),
   userProjects: [] as string[],
   ownership: DefaultOwnership,
};

type PreodersState = typeof initialState;

function filterItems(
   items: IPreorder[],
   projects: Set<string>,
   ownership: OwnershipValue,
   userProjects: string[] | undefined,
): IPreorder[] {
   if (projects.size > 0) {
      return items.filter(i => projects.has(i.project));
   }

   switch (ownership) {
      case OwnershipValue.All: {
         return items;
      }

      case OwnershipValue.My: {
         const userProjectSet = new Set(userProjects);

         return items.filter(i => userProjectSet.has(i.project));
      }
   }
}

function applyFiltersForState(state: PreodersState) {
   state.filteredItems = filterItems(state.allItems, state.projects, state.ownership, state.userProjects);
}

/**
 * Парсит QueryString и возвращает оттуда правильный Ownership
 * Если передано что-то неподдерживаемое, то используется дефолтный
 */
function parseOwnership(valueFromUrl: string | undefined): OwnershipValue {
   const supportedValues = Object.values(OwnershipValue);
   const value = valueFromUrl as OwnershipValue;

   return valueFromUrl && supportedValues.includes(value) ? value : DefaultOwnership;
}

export const preordersSlice = createSlice({
   name: 'preorders',
   initialState,
   reducers: {
      beforeLoading(state) {
         state.allItems = [];
         state.filteredItems = [];
         state.isLoading = true;
      },

      loaded(state, action: PayloadAction<IPreorder[]>) {
         state.allItems = action.payload;
         state.isLoading = false;

         applyFiltersForState(state);
      },

      setLoading(state, action: PayloadAction<boolean>) {
         state.isLoading = action.payload;
      },

      setParamsFromProjects(state, action: PayloadAction<Pick<PreodersState, 'projects' | 'userProjects'>>) {
         state.userProjects = action.payload.userProjects;
         state.projects = action.payload.projects;

         applyFiltersForState(state);
      },

      setParamsFromUrl(state, action: PayloadAction<string>) {
         const urlParams = fromQuery(action.payload) as IPreorderUrlParams;

         state.projects = queryValueToSet(urlParams.project);
         state.ownership = parseOwnership(urlParams.ownership);

         applyFiltersForState(state);
      },
   },
});

export const { beforeLoading, loaded, setLoading, setParamsFromProjects, setParamsFromUrl } = preordersSlice.actions;

export const preordersReducer = preordersSlice.reducer;

//region Async actions
export const loadAll = (): AppThunk => dispatch => {
   dispatch(beforeLoading());

   preorderApi
      .getList()
      .pipe(
         map(listResult => {
            const preorders = listResult.result;
            preorders.sort((a, b) => sortHandler(b.id, a.id));

            return {
               result: preorders,
               total: listResult.total,
            } as IListResult<IPreorder>;
         }),
      )
      .subscribe(
         listResult => dispatch(loaded(listResult.result)),
         err => {
            dispatch(setLoading(false));
            toasts.apiError('Preorder list loading', err);
         },
      );
};
//endregion
