import {
   clearParams,
   DISMISS_REASON,
   FormFieldType,
   FormLayout,
   IFormField,
   IValidationResult,
} from '@yandex-infracloud-ui/libs-next';
import { IModalProps, ModalLayout } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { SyntheticEvent, useEffect, useRef, useState } from 'react';

import { deployConfigField } from '../../../actions/host_actions/actions/commonFields';
import { globalState, IAddPreorderParams, isAddPreorderParamsValid } from '../../../models';
import { ProjectSelectField, RestrictionEditorField, EnumField } from '../../../rich_shared/form_fields';
import { DevJson } from '../../../shared';

const fields: IFormField<IAddPreorderParams>[] = [
   {
      label: 'Preorder ID',
      name: 'id',
      required: true,
      type: FormFieldType.Number,
   },
   {
      component: ProjectSelectField,
      label: 'Project',
      name: 'project',
      required: true,
      type: FormFieldType.Custom,
   },
   {
      checkboxLabel: 'Prepare taken hosts',
      name: 'prepare',
      type: FormFieldType.Boolean,
   },
   {
      ...(deployConfigField as any), // FIXME
      required: false,
      isHidden: p => !p.prepare,
   },
   {
      getHelp: () => 'Restrictions to set for the hosts',
      isHidden: p => !p.prepare,
      label: 'Restrictions',
      name: '_restrictionMode',
      options: [
         { value: 'default', title: 'Take from project' },
         { value: 'manual', title: 'Manual' },
      ],
      type: FormFieldType.Custom,
      component: EnumField,
   },
   {
      component: RestrictionEditorField,
      isHidden: p => !(p.prepare && p._restrictionMode === 'manual'),
      name: 'restrictions',
      type: FormFieldType.Custom,
   },
   {
      label: 'Reason',
      name: 'reason',
      placeholder: 'Reason (optional)',
      type: FormFieldType.Text,
   },
   {
      checkboxLabel: 'Use admin privileges to process the request',
      isHidden: () => (globalState.user.value ? !globalState.user.value.isAdmin : true),
      name: 'sudo',
      type: FormFieldType.Boolean,
   },

   // hidden
   { name: 'deploy_config', isHidden: p => !p.prepare, type: FormFieldType.Hidden },
   { name: 'provisioner', isHidden: p => !p.prepare, type: FormFieldType.Hidden },
];

export const AddPreorderModal = React.memo(({ ok, cancel }: IModalProps<Partial<IAddPreorderParams>>) => {
   // region hooks
   const [validationResult, setValidationResult] = useState<IValidationResult<IAddPreorderParams>>();

   const [value, setValue] = useState<IAddPreorderParams>({
      _restrictionMode: 'default',
      id: 0,
      project: '',
      reason: '',
      restrictions: new Set(),
      sudo: false,
   });

   const cleanedRef = useRef<Partial<IAddPreorderParams>>();

   // endregion

   useEffect(() => {
      const cleaned = clearParams(fields, value) as IAddPreorderParams;
      if (cleaned.prepare && value._restrictionMode === 'manual') {
         cleaned.restrictions = value.restrictions;
      }

      cleanedRef.current = cleaned;
      setValidationResult(isAddPreorderParamsValid(cleaned));
   }, [value]);

   // region handlers
   const confirm = () => {
      if (cleanedRef.current) {
         ok(cleanedRef.current);
      }
   };

   const dismiss = () => cancel(DISMISS_REASON);
   const update = (e: SyntheticEvent | null, v: IAddPreorderParams) => {
      v.deploy_config = v._deployConfig ? v._deployConfig.config : '';
      v.provisioner = v._deployConfig ? v._deployConfig.provisioner : undefined;

      setValue(v);
   };
   // endregion

   // region render
   return (
      <ModalLayout
         title={'Add pre-order'}
         okLabel={'Add'}
         disabled={!validationResult || !validationResult.isValid}
         onOk={confirm}
         onDismiss={dismiss}
      >
         <FormLayout onChange={update} fields={fields} value={value} validationResult={validationResult} />
         <DevJson>{value}</DevJson>
      </ModalLayout>
   );
   // endregion
});

AddPreorderModal.displayName = 'AddPreorderModal';
