import { sortHandler } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';

import styles from './PreorderProgressBar.module.css';

interface IProps {
   failed: number[];
   success: number[];
}

export interface IPart {
   failed: boolean;
   offset: number;
   percent: number;
}

export const PreorderProgressBar = React.memo((props: IProps) => {
   return (
      <svg width={'100%'} height={'8px'} className={styles.svg}>
         {_toParts(props.success, props.failed).map(p => (
            <rect
               key={p.offset}
               width={`${p.percent}%`}
               height={'100%'}
               x={`${p.offset}%`}
               className={p.failed ? styles.failed : styles.success}
            />
         ))}
      </svg>
   );
});

/**
 * Подготавливает список частей для рисования SVG
 *
 * Алгоритм:
 * Для ускорения обрабатываю крайние случаи отдельно (всё хорошо или всё совсем плохо),
 * это тривиальные проверки без итерации.
 *
 * Затем создаю массив частей, и начинаю итерировать по всем записям.
 * Если запись имеет тот же статус успешности, что и текущая часть, то она лишь увеличивается.
 * В противном случае создается новая часть и добавляется в массив частей.
 */
export function _toParts(success: number[], failed: number[]): IPart[] {
   // Если все поломаны
   if (success.length === 0 && failed.length > 0) {
      return [{ failed: true, percent: 100, offset: 0 }];
   }

   // Если ни одна не поломана
   if (failed.length === 0) {
      return [{ failed: false, percent: 100, offset: 0 }];
   }

   const all = [...success, ...failed];
   // Сортировка нужна, чтобы полосочка была прерывистой, а не просто красное в конце
   all.sort((a, b) => sortHandler(a, b));

   // Прочее вычисляем
   const failedSet = new Set(failed); // Для ускорения проверки на ошибочность

   const portion = 100 / all.length;
   let offset = 0;
   let currentPart: IPart = {
      failed: failedSet.has(all[0]),
      offset,
      percent: portion,
   };
   const result: IPart[] = [currentPart];

   // Итерация с 1, т.к. первую запись мы обработали вручную
   for (let i = 1; i < all.length; i += 1) {
      const item = all[i];
      const isFailed = failedSet.has(item);

      if (currentPart.failed === isFailed) {
         currentPart.percent += portion;

         continue;
      }

      offset += currentPart.percent;
      currentPart = {
         failed: isFailed,
         offset,
         percent: portion,
      };
      result.push(currentPart);
   }

   return result;
}
