import {
   classNames,
   DISMISS_REASON,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Formik, FormikHelpers } from 'formik';
import React, { ReactNode, useCallback, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { IApiError, IProject } from '../../../models';
import { projectApi } from '../../../services';
import { LegoButton } from '../../../shared/lego';
import classes from './CloneProjectModal.module.css';
import { InternalForm } from './InternalForm';
import { CloneProjectFormParams, initialValue, validationSchema } from './models';

interface ModalProps extends IModalProps<IProject> {
   children?: ReactNode;
   initialProjectId?: string;
}

export const CloneProjectModal: React.FC<ModalProps> = ({ initialProjectId = null, ok, cancel, children }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const handleFieldChange = useCallback(() => setApiError(''), []);

   const initialValuePatched: CloneProjectFormParams = useMemo(
      () => ({ ...initialValue, projectId: initialProjectId }),
      [initialProjectId],
   );

   const handleSubmit = useCallback(
      (v: CloneProjectFormParams, helpers: FormikHelpers<CloneProjectFormParams>) => {
         const { projectId, ...cloneParams } = v;

         projectApi
            .clone(projectId!, cloneParams)
            .pipe(takeUntil(dismounted))
            .subscribe(
               p => {
                  ok(p);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} title={'Clone project'} showFooter={false} data-e2e={'CloneProjectModal'}>
         <Formik initialValues={initialValuePatched} onSubmit={handleSubmit} validationSchema={validationSchema}>
            {form => (
               <InternalForm onChange={handleFieldChange}>
                  {children}

                  {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                  <Loader visible={form.isSubmitting} inline={true} text={'Project is cloning'} />

                  <div className={classes.spacer} />

                  <LegoButton theme={'clear'} cls={classes.button} onClick={dismiss} disabled={form.isSubmitting}>
                     Cancel
                  </LegoButton>

                  <LegoButton
                     theme={'action'}
                     type={'submit'}
                     cls={classNames(classes.button)}
                     disabled={!form.isValid || form.isSubmitting || form.isValidating}
                  >
                     Clone
                  </LegoButton>
               </InternalForm>
            )}
         </Formik>
      </ModalLayout>
   );
};

CloneProjectModal.displayName = 'CloneProjectModal';
