import { DeclarativeFields, forHumanCapitalized, FormHooks, FormHooksContext } from '@yandex-infracloud-ui/libs';
import { Form, useFormikContext } from 'formik';
import React, { useCallback, useEffect, useMemo, useRef, useState } from 'react';

import { DebugForm } from '../../../shared';
import { idFromName } from '../../../utils';

import classes from './InternalForm.module.css';
import { CloneProjectFormParams, fields } from './models';

interface Props {
   onChange(): void;
}

export const InternalForm: React.FC<Props> = ({ children, onChange }) => {
   const form = useFormikContext<CloneProjectFormParams>();
   const [isIdChangedManually, setIsIdChangedManually] = useState(false);
   const firstRunRef = useRef(true);
   const { setFieldValue } = form;

   const generateIdAndName = useCallback(
      (project: string) => {
         setFieldValue('id', `${project}-clone`);
         setFieldValue('name', forHumanCapitalized(project as string) + ' clone');
         setIsIdChangedManually(false);
      },
      [setFieldValue],
   );

   useEffect(() => {
      if (firstRunRef.current && form.values.projectId !== null) {
         generateIdAndName(form.values.projectId);
      }
      firstRunRef.current = false;
   }, [form.values.projectId, generateIdAndName]);

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange(field, value) {
               onChange();

               switch (field) {
                  case 'name': {
                     if (!isIdChangedManually) {
                        setFieldValue('id', idFromName(value as string));
                     }
                     break;
                  }

                  case 'id': {
                     setIsIdChangedManually(true);
                     break;
                  }

                  case 'projectId': {
                     generateIdAndName(value as string);
                     break;
                  }
               }
            },
         } as FormHooks<CloneProjectFormParams>),
      [generateIdAndName, isIdChangedManually, onChange, setFieldValue],
   );

   return (
      <Form className={classes.form}>
         <FormHooksContext.Provider value={formHooks}>
            <DeclarativeFields configs={fields} />
         </FormHooksContext.Provider>

         <footer className={classes.footer}>{children}</footer>

         <DebugForm />
      </Form>
   );
};

InternalForm.displayName = 'InternalForm';
