import {
   classNames,
   DISMISS_REASON,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Formik, FormikHelpers } from 'formik';
import React, { PropsWithChildren, useCallback, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { IApiError, IProject } from '../../../models';
import { config, projectApi } from '../../../services';
import { LegoButton } from '../../../shared/lego';

import { InternalForm } from './InternalForm';
import {
   formParamsToNewProject,
   initialValue,
   NewProjectShortFormParams,
   validationSchemaInternal,
   validationSchemaExternal,
} from './models';
import classes from './NewProjectModal.module.css';

export const NewProjectModal: React.FC<PropsWithChildren<IModalProps<IProject>>> = ({ ok, cancel, children }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');

   const handleFieldChange = useCallback(() => setApiError(''), []);

   const handleSubmit = useCallback(
      (v: NewProjectShortFormParams, helpers: FormikHelpers<NewProjectShortFormParams>) => {
         projectApi
            .fastCreate(formParamsToNewProject(v))
            .pipe(takeUntil(dismounted))
            .subscribe(
               p => {
                  ok(p);
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, ok],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} title={'Create new project'} showFooter={false}>
         <Formik
            initialValues={initialValue}
            onSubmit={handleSubmit}
            validationSchema={config.isExternal ? validationSchemaExternal : validationSchemaInternal}
         >
            {form => (
               <InternalForm onChange={handleFieldChange}>
                  {children}

                  {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                  <Loader visible={form.isSubmitting} inline={true} text={'Project is creating'} />

                  <div className={classes.spacer} />

                  <LegoButton theme={'clear'} cls={classes.button} onClick={dismiss} disabled={form.isSubmitting}>
                     Cancel
                  </LegoButton>

                  <LegoButton
                     theme={'action'}
                     type={'submit'}
                     cls={classNames(classes.button)}
                     disabled={!form.isValid || form.isSubmitting || form.isValidating}
                  >
                     Create
                  </LegoButton>
               </InternalForm>
            )}
         </Formik>
      </ModalLayout>
   );
};

NewProjectModal.displayName = 'NewProjectModal';
